#!/bin/bash
# =============================================================================
# Book Assembly Script for Research Development Framework
# =============================================================================
# Assembles markdown chapter files into a complete manuscript
#
# Usage:
#   ./assemble_book.sh                          # Assemble with defaults
#   ./assemble_book.sh --title "My Book"        # Custom title
#   ./assemble_book.sh --output mybook.md       # Custom output filename
#   ./assemble_book.sh --preserve-frontmatter   # Keep YAML as editor's notes
#   ./assemble_book.sh --bibliography refs.bib  # Include BibTeX bibliography
# =============================================================================

set -e

# Default configuration
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
CHAPTERS_DIR="${SCRIPT_DIR}/chapters"
OUTPUT_DIR="${SCRIPT_DIR}/output"
BOOK_TITLE="Research Manuscript"
OUTPUT_FILENAME="complete_manuscript.md"
PRESERVE_FRONTMATTER=false
BIBLIOGRAPHY_FILE=""

# Parse command line arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --title)
            BOOK_TITLE="$2"
            shift 2
            ;;
        --output)
            OUTPUT_FILENAME="$2"
            shift 2
            ;;
        --chapters)
            CHAPTERS_DIR="$2"
            shift 2
            ;;
        --preserve-frontmatter)
            PRESERVE_FRONTMATTER=true
            shift
            ;;
        --bibliography)
            BIBLIOGRAPHY_FILE="$2"
            shift 2
            ;;
        --help)
            echo "Usage: $0 [OPTIONS]"
            echo ""
            echo "Options:"
            echo "  --title TITLE           Book title (default: 'Research Manuscript')"
            echo "  --output FILE           Output filename (default: 'complete_manuscript.md')"
            echo "  --chapters DIR          Chapter directory (default: './chapters')"
            echo "  --preserve-frontmatter  Convert YAML front matter to Editor's Notes"
            echo "  --bibliography FILE     BibTeX file for citation processing"
            echo "  --help                  Show this help message"
            echo ""
            echo "Examples:"
            echo "  # Basic assembly"
            echo "  ./assemble_book.sh --title 'My Research'"
            echo ""
            echo "  # Keep source metadata as footnotes"
            echo "  ./assemble_book.sh --title 'My Book' --preserve-frontmatter"
            echo ""
            echo "  # With bibliography for citation processing"
            echo "  ./assemble_book.sh --bibliography refs.bib --title 'My Book'"
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            exit 1
            ;;
    esac
done

# Ensure directories exist
mkdir -p "$OUTPUT_DIR"

# Check for chapter files
if [ ! -d "$CHAPTERS_DIR" ]; then
    echo "Error: Chapters directory not found: $CHAPTERS_DIR"
    exit 1
fi

CHAPTER_FILES=$(find "$CHAPTERS_DIR" -name "*.md" -type f | sort)

if [ -z "$CHAPTER_FILES" ]; then
    echo "Error: No markdown files found in $CHAPTERS_DIR"
    exit 1
fi

# Count chapters
CHAPTER_COUNT=$(echo "$CHAPTER_FILES" | wc -l)
echo "Found $CHAPTER_COUNT chapter files"

# Output file path
OUTPUT_FILE="${OUTPUT_DIR}/${OUTPUT_FILENAME}"

# Start building the manuscript
echo "Assembling manuscript: $BOOK_TITLE"
echo ""

# Create title page
cat > "$OUTPUT_FILE" << EOF
---
title: "${BOOK_TITLE}"
date: "$(date +%Y-%m-%d)"
---

# ${BOOK_TITLE}

---

EOF

# Append each chapter
CURRENT=0
for chapter in $CHAPTER_FILES; do
    CURRENT=$((CURRENT + 1))
    CHAPTER_NAME=$(basename "$chapter" .md)

    echo "  [$CURRENT/$CHAPTER_COUNT] Adding: $CHAPTER_NAME"

    # Add page break before each chapter (except first)
    if [ $CURRENT -gt 1 ]; then
        echo -e "\n\n---\n\n" >> "$OUTPUT_FILE"
    fi

    # Extract content (handle YAML front matter)
    if head -1 "$chapter" | grep -q "^---$"; then
        # Has front matter
        if [ "$PRESERVE_FRONTMATTER" = true ]; then
            # Extract front matter and convert to Editor's Note
            FRONTMATTER=$(awk '/^---$/{p++} p==1 && !/^---$/' "$chapter")

            # Parse key fields from YAML
            FM_TITLE=$(echo "$FRONTMATTER" | grep -E "^title:" | sed 's/title:\s*//' | tr -d '"')
            FM_SOURCE=$(echo "$FRONTMATTER" | grep -E "^source:" | sed 's/source:\s*//' | tr -d '"')
            FM_DATE=$(echo "$FRONTMATTER" | grep -E "^date:" | sed 's/date:\s*//' | tr -d '"')
            FM_AUTHOR=$(echo "$FRONTMATTER" | grep -E "^author:" | sed 's/author:\s*//' | tr -d '"')

            # Add Editor's Note if we found metadata
            if [ -n "$FM_TITLE" ] || [ -n "$FM_SOURCE" ] || [ -n "$FM_AUTHOR" ]; then
                echo "" >> "$OUTPUT_FILE"
                echo "> **Editor's Note:** This chapter is derived from" >> "$OUTPUT_FILE"
                if [ -n "$FM_TITLE" ]; then
                    echo -n "> *${FM_TITLE}*" >> "$OUTPUT_FILE"
                fi
                if [ -n "$FM_AUTHOR" ]; then
                    echo -n " by ${FM_AUTHOR}" >> "$OUTPUT_FILE"
                fi
                if [ -n "$FM_DATE" ]; then
                    echo -n " (${FM_DATE})" >> "$OUTPUT_FILE"
                fi
                echo "." >> "$OUTPUT_FILE"
                if [ -n "$FM_SOURCE" ]; then
                    echo "> Source: ${FM_SOURCE}" >> "$OUTPUT_FILE"
                fi
                echo "" >> "$OUTPUT_FILE"
            fi

            # Add chapter content (skip front matter)
            awk '/^---$/{p++} p==2{p++} p>2' "$chapter" >> "$OUTPUT_FILE"
        else
            # Skip front matter entirely
            awk '/^---$/{p++} p==2{p++} p>2' "$chapter" >> "$OUTPUT_FILE"
        fi
    else
        # No front matter - include everything
        cat "$chapter" >> "$OUTPUT_FILE"
    fi

    echo "" >> "$OUTPUT_FILE"
done

# Calculate word count
WORD_COUNT=$(wc -w < "$OUTPUT_FILE")

echo ""
echo "=============================================="
echo "ASSEMBLY COMPLETE"
echo "=============================================="
echo "Title:       $BOOK_TITLE"
echo "Chapters:    $CHAPTER_COUNT"
echo "Word Count:  $WORD_COUNT"
echo "Output:      $OUTPUT_FILE"
echo "=============================================="

# Generate DOCX if pandoc is available
if command -v pandoc &> /dev/null; then
    DOCX_FILE="${OUTPUT_DIR}/${OUTPUT_FILENAME%.md}.docx"
    echo ""
    echo "Generating DOCX with Pandoc..."

    # Build pandoc command with optional bibliography
    PANDOC_OPTS=(
        "$OUTPUT_FILE"
        -f "markdown+smart"
        -t "docx"
        -o "$DOCX_FILE"
        --standalone
        --highlight-style=tango
        --toc
        --toc-depth=2
    )

    # Add bibliography processing if file specified
    if [ -n "$BIBLIOGRAPHY_FILE" ] && [ -f "$BIBLIOGRAPHY_FILE" ]; then
        echo "Using bibliography: $BIBLIOGRAPHY_FILE"
        PANDOC_OPTS+=(--bibliography="$BIBLIOGRAPHY_FILE" --citeproc)
    fi

    pandoc "${PANDOC_OPTS[@]}"
    echo "DOCX created: $DOCX_FILE"

    # Generate EPUB
    EPUB_FILE="${OUTPUT_DIR}/${OUTPUT_FILENAME%.md}.epub"
    PANDOC_EPUB_OPTS=(
        "$OUTPUT_FILE"
        -f "markdown+smart"
        -t "epub3"
        -o "$EPUB_FILE"
        --standalone
        --toc
        --toc-depth=2
    )

    if [ -n "$BIBLIOGRAPHY_FILE" ] && [ -f "$BIBLIOGRAPHY_FILE" ]; then
        PANDOC_EPUB_OPTS+=(--bibliography="$BIBLIOGRAPHY_FILE" --citeproc)
    fi

    pandoc "${PANDOC_EPUB_OPTS[@]}"
    echo "EPUB created: $EPUB_FILE"

    # Generate PDF if pdflatex is available
    if command -v pdflatex &> /dev/null; then
        PDF_FILE="${OUTPUT_DIR}/${OUTPUT_FILENAME%.md}.pdf"
        PANDOC_PDF_OPTS=(
            "$OUTPUT_FILE"
            -f "markdown+smart"
            -o "$PDF_FILE"
            --standalone
            --toc
            --toc-depth=2
            --pdf-engine=pdflatex
        )

        if [ -n "$BIBLIOGRAPHY_FILE" ] && [ -f "$BIBLIOGRAPHY_FILE" ]; then
            PANDOC_PDF_OPTS+=(--bibliography="$BIBLIOGRAPHY_FILE" --citeproc)
        fi

        pandoc "${PANDOC_PDF_OPTS[@]}" 2>/dev/null && echo "PDF created: $PDF_FILE" || echo "PDF skipped (LaTeX error)"
    fi
else
    echo ""
    echo "Note: Install pandoc to auto-generate DOCX, EPUB, and PDF"
    echo "      Run: python format_docx.py to convert manually"
fi

echo ""
echo "Done!"
