# QA Test Suite - Research Development Framework

Comprehensive testing suite for the Research Development Framework.

## Quick Start

```bash
cd /var/www/html/research/Research_development/QA_TEST

# Run quick smoke tests (no API key needed)
python run_tests.py --quick

# Run full tests without cloud features
python run_tests.py

# Run full tests with OpenAI API
python run_tests.py --api-key YOUR_KEY

# Generate HTML report
python run_tests.py --api-key YOUR_KEY --report

# Run specific test modules
python run_tests.py --module v4         # CLI tests
python run_tests.py --module database   # Database tests
python run_tests.py --module workflow   # Book research workflow tests
python run_tests.py --module rag        # All Advanced RAG tests
```

## Test Modules

### Core Tests
| Module | Tests | Requires API |
|--------|-------|--------------|
| Configuration | Config files, paths, environment | No |
| Libraries | All required Python packages | No |
| Database | Connection, tables, pgvector | No |
| Pipeline | Taxonomist, Clusterer, Chunker | No |
| Statistical Classification | RAKE/TF-IDF keyword extraction | No |
| Statistical Clustering | TF-IDF vectorization | No |
| Document Processing | Text extraction, chunking | No |
| API Endpoints | All REST endpoints | No |
| Embeddings | OpenAI embedding generation | **Yes** |

### CLI Tests
| Module | Tests | Requires API |
|--------|-------|--------------|
| rdf_cli | `rdf --help`, `rdf commands` | No |
| rdf_search | `rdf search` command | No |
| rdf_fetch | `rdf fetch` command | No |
| rdf_diff | `rdf diff` command | No |
| rdf_assess | `rdf assess` command | No |
| rdf_status | `rdf status` command | No |
| rdf_health | `rdf health` command | No |

### Advanced RAG Tests
| Module | Tests | Requires API |
|--------|-------|--------------|
| reranking | Cross-encoder re-ranking | No |
| graphrag | GraphRAG knowledge graph retrieval | No |
| semantic_chunking | Embedding-based chunking | No |
| research_agent | Autonomous research agent | Optional |
| rag_comparison | RAG comparison tests | No |

### Feature Tests
| Module | Tests | Requires API |
|--------|-------|--------------|
| citation | Citation key generation and freezing | No |
| pinning | Document pinning for prioritization | No |
| gaps | Research gap tracking and pinning | No |
| context | Smart context selection for LLM | No |
| semantic_graph | Semantic graph traversal | No |
| workflow | Book research workflow (5 phases) | No |

## Command Line Options

```
python run_tests.py [options]

Options:
  --quick           Run only quick smoke tests
  --module NAME     Run specific test module
  --report          Generate HTML report in reports/
  --api-key KEY     Provide OpenAI API key directly
```

## Test Reports

HTML reports are saved in `reports/` with timestamps:
```
reports/
  test_report_20251212_092455.html
  ...
```

## Expected Results

With full setup and API key:
- **100+ tests** should pass
- **Some skips** expected if:
  - No documents/chunks in database
  - sentence-transformers not installed
- **0 failures** expected

## Fixing Common Issues

### "No module named 'flask'"
```bash
pip install flask
```

### "No module named 'sklearn'"
```bash
pip install scikit-learn
```

### "NLTK punkt not found"
```bash
python -c "import nltk; nltk.download('punkt'); nltk.download('punkt_tab'); nltk.download('stopwords')"
```

### "rdf: command not found"
```bash
chmod +x ../rdf
```

## Test Coverage

The test suite covers:

1. **Configuration**
   - YAML config loading
   - Environment files
   - Path configuration

2. **Database**
   - PostgreSQL connection
   - Core tables (documents, chunks, concepts, topics)
   - pgvector extension

3. **Pipeline Components**
   - DocumentProcessor
   - TextChunker
   - Taxonomist (statistical mode)
   - SemanticClusterer

4. **CLI Commands**
   - `rdf` entrypoint
   - All 17 commands
   - JSON output format
   - Autonomy levels

5. **Advanced RAG**
   - Cross-encoder re-ranking
   - GraphRAG knowledge graph retrieval
   - Semantic chunking

6. **Writing Workflows**
   - Book workflow phases
   - Essay generation
   - Outline creation
   - Validation and polish

## Adding New Tests

Tests are organized in `run_tests.py` as functions:

```python
def test_my_feature(results: TestResults):
    print_header("MY FEATURE TESTS")

    try:
        # Test code here
        results.add_pass("Test name", "Optional message")
    except Exception as e:
        results.add_fail("Test name", str(e))
```

Then add to module_map in `run_all_tests()`:
```python
module_map = {
    # ...
    'my_feature': test_my_feature,
}
```

## Test Modules Notes

The following test modules are deprecated:
- `sessions` - Interactive sessions removed (use resume tokens)
- `rules_tui` - TUI removed (use CLI)

CLI test modules:
- `v4` / `rdf_cli` - Unified CLI tests
- `rdf_*` - Individual command tests
