#!/usr/bin/env python3
"""
RDF Write - Universal entry point for writing tasks

Infers essay vs book from parameters and dispatches to appropriate workflow.

Usage:
    rdf write "Topic" --pages 20              # → essay
    rdf write "Topic" --chapters 5            # → book
    rdf write "Topic" --type essay --pages 20
    rdf write "Topic" --type book --chapters 5 --pages 100
"""

import argparse
import json
import sys
import subprocess
from pathlib import Path

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))

from pipeline.cli_utils import success_response, error_response, ErrorCodes


def infer_type(chapters: int, pages: int, explicit_type: str) -> str:
    """Infer whether to create essay or book."""
    if explicit_type:
        return explicit_type

    if chapters and chapters > 1:
        return "book"

    # Default to essay for simpler requests
    return "essay"


def main():
    parser = argparse.ArgumentParser(
        description="Universal entry point for writing tasks",
        formatter_class=argparse.RawDescriptionHelpFormatter
    )
    parser.add_argument("topic", help="Topic to write about")
    parser.add_argument("--type", choices=["essay", "book"], default=None,
                        help="Explicit type (inferred if not set)")
    parser.add_argument("--pages", type=int, default=None,
                        help="Target page count")
    parser.add_argument("--chapters", type=int, default=None,
                        help="Number of chapters (implies book)")
    parser.add_argument("--style", default="accessible",
                        choices=["academic", "accessible", "narrative", "popular"],
                        help="Writing style")
    parser.add_argument("--autonomy", default=None,
                        choices=["full", "supervised", "interactive"],
                        help="Autonomy level (default: full for essay, supervised for book)")
    parser.add_argument("--evidence-first", action="store_true",
                        help="Require evidence before claims")
    parser.add_argument("--strict", action="store_true",
                        help="Library sources only")
    parser.add_argument("--format", choices=["json", "text"], default="json",
                        help="Output format")

    args = parser.parse_args()

    # Infer type
    write_type = infer_type(args.chapters, args.pages, args.type)

    # Set defaults based on type
    if write_type == "essay":
        pages = args.pages or 10
        autonomy = args.autonomy or "full"

        # Build essay command
        cmd = [
            sys.executable,
            str(Path(__file__).parent / "essay.py"),
            args.topic,
            "--pages", str(pages),
            "--style", args.style,
            "--autonomy", autonomy,
            "--format", args.format
        ]

        if args.evidence_first:
            cmd.append("--evidence-first")
        if args.strict:
            cmd.append("--strict")

    else:  # book
        chapters = args.chapters or 5
        pages = args.pages or (chapters * 20)
        autonomy = args.autonomy or "supervised"

        # For books, we first generate outline then run book workflow
        # Step 1: Generate outline
        outline_cmd = [
            sys.executable,
            str(Path(__file__).parent / "outline.py"),
            args.topic,
            "--chapters", str(chapters),
            "--pages", str(pages),
            "--style", args.style,
            "--format", "json"
        ]

        try:
            result = subprocess.run(outline_cmd, capture_output=True, text=True)
            if result.returncode != 0:
                print(result.stdout)
                print(result.stderr, file=sys.stderr)
                return result.returncode

            # Parse outline result to get path
            outline_result = json.loads(result.stdout)
            outline_path = outline_result.get("data", {}).get("outline_path")

            if not outline_path:
                resp = error_response(
                    ErrorCodes.UNKNOWN_ERROR,
                    "Failed to generate outline"
                )
                resp.print_json()
                return 1

            # Step 2: Run book workflow
            cmd = [
                sys.executable,
                str(Path(__file__).parent / "book.py"),
                outline_path,
                "--autonomy", autonomy,
                "--format", args.format
            ]

            if args.strict:
                cmd.append("--strict-library-only")

        except json.JSONDecodeError:
            resp = error_response(
                ErrorCodes.UNKNOWN_ERROR,
                "Failed to parse outline result"
            )
            resp.print_json()
            return 1

    # Execute the command
    try:
        result = subprocess.run(cmd)
        return result.returncode
    except Exception as e:
        resp = error_response(
            ErrorCodes.UNKNOWN_ERROR,
            str(e)
        )
        resp.print_json()
        return 1


if __name__ == "__main__":
    sys.exit(main())
