#!/usr/bin/env python3
"""
Research Workflow CLI

CLI-first research workflow for book projects, optimized for Claude Code orchestration.

Usage:
    # Full workflow from book outline
    python research_workflow.py book_outline.yaml

    # From subjects file
    python research_workflow.py --subjects-file subjects.txt --title "My Research"

    # From command-line subjects
    python research_workflow.py --subjects "mystery schools" "alchemy" --title "Quick Study"

    # Resume interrupted workflow
    python research_workflow.py --resume BOOK_xxx

    # Run specific phase only
    python research_workflow.py book_outline.yaml --phase 2

    # Auto-fill gaps with Tavily (budget limited)
    python research_workflow.py --resume BOOK_xxx --phase 3 --auto-fill-gaps --tavily-budget 100

    # Generate chapter drafts
    python research_workflow.py --resume BOOK_xxx --phase 5 --generate-drafts

    # Dry run preview
    python research_workflow.py book_outline.yaml --dry-run

    # Machine-readable status for Claude Code
    python research_workflow.py --status BOOK_xxx --format json

    # List all projects
    python research_workflow.py --list-projects
"""

import argparse
import json
import logging
import sys
from pathlib import Path
from datetime import datetime

# Setup path for imports
sys.path.insert(0, str(Path(__file__).parent))

from config import LOGGING_CONFIG, BASE_DIR, BOOK_WORKFLOW_CONFIG

LOG_LEVEL = LOGGING_CONFIG.get('level', 'INFO')

# Book workflow imports
from book_workflow_models import BookProject, WorkflowCheckpoint, list_book_projects
from book_workflow_input import (
    parse_yaml_outline,
    parse_json_outline,
    parse_subjects_file,
    parse_subjects_list,
    parse_outline_file,
    validate_project,
)
from book_workflow_output import (
    generate_gaps_markdown,
    parse_gaps_markdown,
    generate_research_summary,
    generate_sources_json,
    generate_status_json,
    write_all_outputs,
)
from book_workflow_phases import (
    Phase1Executor,
    Phase2Executor,
    Phase3Executor,
    Phase4Executor,
    Phase5Executor,
    run_workflow,
)

# Setup logging
logging.basicConfig(
    level=getattr(logging, LOG_LEVEL, logging.INFO),
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# Default projects directory
PROJECTS_DIR = BOOK_WORKFLOW_CONFIG.get('projects_dir', BASE_DIR / 'book_projects')


# =============================================================================
# CLI ARGUMENT PARSER
# =============================================================================

def create_parser() -> argparse.ArgumentParser:
    """Create the CLI argument parser."""
    parser = argparse.ArgumentParser(
        prog='research_workflow',
        description='CLI-first research workflow for book projects',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Start from YAML outline
  python research_workflow.py outline.yaml

  # Start from subjects
  python research_workflow.py --subjects "alchemy" "hermeticism" --title "Esoteric Study"

  # Resume with auto-fill gaps
  python research_workflow.py --resume BOOK_xxx --phase 3 --auto-fill-gaps

  # Check status (JSON for Claude Code)
  python research_workflow.py --status BOOK_xxx --format json

For Claude Code orchestration, use --format json for machine-readable output.
        """
    )

    # Input source (mutually exclusive)
    input_group = parser.add_argument_group('Input Options')
    input_group.add_argument(
        'outline_file',
        nargs='?',
        type=str,
        help='Path to YAML/JSON book outline file'
    )
    input_group.add_argument(
        '--subjects-file',
        type=str,
        help='Path to text file with subjects (one per line)'
    )
    input_group.add_argument(
        '--subjects',
        nargs='+',
        type=str,
        help='List of subjects from command line'
    )
    input_group.add_argument(
        '--title',
        type=str,
        default='Research Project',
        help='Project title (used with --subjects or --subjects-file)'
    )

    # Resume/status options
    control_group = parser.add_argument_group('Project Control')
    control_group.add_argument(
        '--resume',
        type=str,
        metavar='PROJECT_ID',
        help='Resume an existing project by ID'
    )
    control_group.add_argument(
        '--status',
        type=str,
        metavar='PROJECT_ID',
        help='Show status of a project'
    )
    control_group.add_argument(
        '--list-projects',
        action='store_true',
        help='List all book projects'
    )
    control_group.add_argument(
        '--context',
        type=str,
        metavar='PROJECT_ID',
        help='Output Claude-optimized project context for session resumption'
    )

    # Phase control
    phase_group = parser.add_argument_group('Phase Control')
    phase_group.add_argument(
        '--phase',
        type=int,
        choices=[1, 2, 3, 4, 5],
        help='Run specific phase only (1=Research, 2=Gap Analysis, 3=Gap Fill, 4=Synthesis, 5=Draft)'
    )
    phase_group.add_argument(
        '--stop-after-phase',
        type=int,
        choices=[1, 2, 3, 4, 5],
        help='Stop after completing this phase'
    )

    # Gap filling options
    gaps_group = parser.add_argument_group('Gap Filling Options')
    gaps_group.add_argument(
        '--auto-fill-gaps',
        action='store_true',
        help='Automatically fill gaps with Tavily web search'
    )
    gaps_group.add_argument(
        '--tavily-budget',
        type=int,
        default=50,
        help='Maximum Tavily credits to use (default: 50)'
    )
    gaps_group.add_argument(
        '--gaps-file',
        type=str,
        help='Path to edited gaps.md file (for manual gap selection)'
    )
    gaps_group.add_argument(
        '--approve-all-gaps',
        action='store_true',
        help='Auto-approve all gaps for filling (skip gaps.md review)'
    )
    gaps_group.add_argument(
        '--approve-priority',
        type=str,
        choices=['high', 'medium', 'low', 'all'],
        help='Auto-approve gaps of this priority or higher'
    )
    gaps_group.add_argument(
        '--approve-gaps',
        type=str,
        help='Comma-separated list of gap indices to approve (e.g., "1,3,5")'
    )

    # Draft generation
    draft_group = parser.add_argument_group('Draft Generation')
    draft_group.add_argument(
        '--generate-drafts',
        action='store_true',
        help='Generate chapter drafts in Phase 5'
    )
    draft_group.add_argument(
        '--draft-style',
        type=str,
        choices=['academic', 'narrative', 'reference'],
        default='academic',
        help='Style for generated drafts (default: academic)'
    )

    # Research options
    research_group = parser.add_argument_group('Research Options')
    research_group.add_argument(
        '--max-iterations',
        type=int,
        default=5,
        help='Max research iterations per subject (default: 5)'
    )
    research_group.add_argument(
        '--no-graphrag',
        action='store_true',
        help='Disable GraphRAG (use only vector search)'
    )
    research_group.add_argument(
        '--no-rerank',
        action='store_true',
        help='Disable reranking'
    )

    # Output options
    output_group = parser.add_argument_group('Output Options')
    output_group.add_argument(
        '--format',
        type=str,
        choices=['text', 'json'],
        default='text',
        help='Output format (default: text, use json for Claude Code)'
    )
    output_group.add_argument(
        '--output-dir',
        type=str,
        help='Custom output directory'
    )
    output_group.add_argument(
        '--dry-run',
        action='store_true',
        help='Preview workflow without executing'
    )
    output_group.add_argument(
        '--verbose', '-v',
        action='store_true',
        help='Enable verbose output'
    )
    output_group.add_argument(
        '--quiet', '-q',
        action='store_true',
        help='Minimal output (errors only)'
    )

    return parser


# =============================================================================
# COMMAND HANDLERS
# =============================================================================

def handle_list_projects(args) -> int:
    """List all book projects."""
    projects = list_book_projects(PROJECTS_DIR)

    if args.format == 'json':
        print(json.dumps(projects, indent=2))
        return 0

    if not projects:
        print("No book projects found.")
        print(f"Projects directory: {PROJECTS_DIR}")
        return 0

    print(f"\n{'='*60}")
    print("BOOK RESEARCH PROJECTS")
    print(f"{'='*60}\n")

    for proj in projects:
        status_emoji = {
            'in_progress': '🔄',
            'completed': '✅',
            'paused': '⏸️',
        }.get(proj['status'], '❓')

        print(f"{status_emoji} {proj['project_id']}")
        print(f"   Title: {proj['title']}")
        print(f"   Phase: {proj['current_phase']}/5")
        print(f"   Chapters: {proj['chapter_count']}")
        print(f"   Created: {proj['created_at']}")
        print(f"   Directory: {proj['directory']}")
        print()

    return 0


def handle_status(args) -> int:
    """Show status of a specific project."""
    project_id = args.status

    # Find project directory
    project_dir = PROJECTS_DIR / project_id
    if not project_dir.exists():
        # Try to find by partial match
        matches = list(PROJECTS_DIR.glob(f"*{project_id}*"))
        if len(matches) == 1:
            project_dir = matches[0]
        elif len(matches) > 1:
            print(f"Error: Multiple projects match '{project_id}':")
            for m in matches:
                print(f"  - {m.name}")
            return 1
        else:
            print(f"Error: Project not found: {project_id}")
            return 1

    # Load project
    try:
        project = BookProject.load(project_dir)
    except Exception as e:
        print(f"Error loading project: {e}")
        return 1

    if args.format == 'json':
        print(generate_status_json(project))
        return 0

    # Text format status
    status = project.get_status_summary()

    print(f"\n{'='*60}")
    print(f"PROJECT STATUS: {project.title}")
    print(f"{'='*60}\n")

    print(f"Project ID: {project.project_id}")
    print(f"Status: {project.status}")
    print(f"Current Phase: {project.current_phase}/5")
    print()

    print("PHASES:")
    phase_names = {
        1: "Initial Research",
        2: "Gap Analysis",
        3: "Gap Filling",
        4: "Synthesis",
        5: "Draft Generation"
    }
    for phase, name in phase_names.items():
        phase_status = project.phase_status.get(phase, 'pending')
        emoji = {'completed': '✅', 'in_progress': '🔄', 'pending': '⏳'}.get(phase_status, '❓')
        marker = " <-- current" if phase == project.current_phase else ""
        print(f"  {emoji} Phase {phase}: {name} ({phase_status}){marker}")
    print()

    print("PROGRESS:")
    print(f"  Chapters: {status['chapters']['completed']}/{status['chapters']['total']}")
    print(f"  Subjects researched: {project.completed_subjects}/{project.total_subjects}")
    print()

    print("GAPS:")
    print(f"  Total: {status['gaps']['total']}")
    print(f"  Filled: {status['gaps']['filled']}")
    print(f"  Unfilled: {status['gaps']['unfilled']}")
    print()

    if project.tavily_credits_used > 0:
        print("TAVILY CREDITS:")
        print(f"  Used: {project.tavily_credits_used}")
        print(f"  Budget: {project.tavily_credit_budget}")
        print()

    # Next action suggestion
    if status.get('next_action'):
        print(f"NEXT ACTION: {status['next_action']}")
        if status.get('command'):
            print(f"  Command: {status['command']}")

    return 0


def handle_context(args) -> int:
    """Output Claude-optimized project context for session resumption."""
    project_id = args.context

    # Find project directory
    project_dir = PROJECTS_DIR / project_id
    if not project_dir.exists():
        # Try to find by partial match
        matches = list(PROJECTS_DIR.glob(f"*{project_id}*"))
        if len(matches) == 1:
            project_dir = matches[0]
        elif len(matches) > 1:
            print(f"Error: Multiple projects match '{project_id}':")
            for m in matches:
                print(f"  - {m.name}")
            return 1
        else:
            print(f"Error: Project not found: {project_id}")
            return 1

    # Load project
    try:
        project = BookProject.load(project_dir / 'project.json')
    except Exception as e:
        print(f"Error loading project: {e}")
        return 1

    # Generate Claude-optimized context
    context = generate_claude_context(project, project_dir)
    print(context)
    return 0


def generate_claude_context(project: BookProject, project_dir: Path) -> str:
    """Generate Claude-optimized project context for session resumption."""
    lines = [
        f"## Project: {project.title}",
        "",
        "### Status",
        f"- **Project ID:** `{project.project_id}`",
        f"- **Phase:** {project.current_phase}/5 ({get_phase_name(project.current_phase)})",
        f"- **Chapters:** {len(project.chapters)} total",
        "",
    ]

    # Chapter summary
    lines.append("### Chapters")
    lines.append("")
    for chapter in project.chapters:
        completed = len(chapter.get_completed_subjects())
        total = len(chapter.subjects)
        if chapter.status == 'complete':
            status_icon = "COMPLETE"
        elif chapter.status in ['researching', 'synthesizing', 'drafting']:
            status_icon = "IN PROGRESS"
        else:
            status_icon = "PENDING"

        lines.append(f"{chapter.chapter_number}. **{chapter.title}** ({status_icon})")
        lines.append(f"   - Subjects: {completed}/{total} researched")

        # Show current subject if in progress
        pending = chapter.get_pending_subjects()
        if pending and chapter.status == 'researching':
            lines.append(f"   - Currently researching: \"{pending[0].subject}\"")

        # Show gap count
        gaps = chapter.get_unfilled_gaps()
        if gaps:
            lines.append(f"   - Gaps remaining: {len(gaps)}")

    lines.append("")

    # Recent activity (from session log if exists)
    session_log_path = project_dir / 'session_log.json'
    if session_log_path.exists():
        try:
            with open(session_log_path, 'r') as f:
                session_log = json.load(f)
            if session_log.get('recent_activity'):
                lines.append("### Recent Activity")
                lines.append("")
                for activity in session_log['recent_activity'][-5:]:
                    lines.append(f"- {activity}")
                lines.append("")
        except Exception:
            pass

    # Key findings (if any synthesis exists)
    findings = []
    for chapter in project.chapters:
        if chapter.research_summary:
            # Extract first key point
            summary_lines = chapter.research_summary.split('\n')
            for line in summary_lines[:3]:
                if line.strip() and not line.startswith('#'):
                    findings.append(f"Ch.{chapter.chapter_number}: {line.strip()[:100]}")
                    break

    if findings:
        lines.append("### Key Findings")
        lines.append("")
        for finding in findings[:5]:
            lines.append(f"- {finding}")
        lines.append("")

    # Open questions / gaps
    all_unfilled = [g for g in project.all_gaps if not g.get('searched', False)]
    if all_unfilled:
        lines.append("### Open Questions (Unfilled Gaps)")
        lines.append("")
        for gap in all_unfilled[:5]:
            lines.append(f"- {gap.get('description', gap.get('suggested_query', 'Unknown'))[:80]}")
        if len(all_unfilled) > 5:
            lines.append(f"- *... and {len(all_unfilled) - 5} more gaps*")
        lines.append("")

    # Quick commands
    lines.append("### Quick Commands")
    lines.append("")
    lines.append(f"```bash")
    lines.append(f"# Continue workflow")
    lines.append(f"python3 pipeline/research_workflow.py --resume {project.project_id}")
    lines.append(f"")
    lines.append(f"# Check status")
    lines.append(f"python3 pipeline/research_workflow.py --status {project.project_id}")
    lines.append(f"")
    lines.append(f"# Search library")
    lines.append(f"python3 pipeline/search_export.py \"your query\" --top 10")
    lines.append(f"")
    lines.append(f"# Validate a draft")
    lines.append(f"python3 pipeline/validate_draft.py chapter.md")
    lines.append(f"```")
    lines.append("")

    # Suggested next action
    lines.append("### Suggested Next Action")
    lines.append("")
    if project.current_phase == 1:
        pending_chapters = [c for c in project.chapters if c.status == 'pending']
        if pending_chapters:
            lines.append(f"Continue Phase 1 research. Next chapter: **{pending_chapters[0].title}**")
        else:
            lines.append("Phase 1 complete. Ready to run Phase 2 (Gap Analysis).")
    elif project.current_phase == 2:
        lines.append(f"Review `{project_dir}/gaps.md` and decide which gaps to fill.")
        lines.append("Then run Phase 3 with `--auto-fill-gaps` or manual selection.")
    elif project.current_phase == 3:
        if all_unfilled:
            lines.append(f"Fill remaining {len(all_unfilled)} gaps, then run Phase 4 (Synthesis).")
        else:
            lines.append("All gaps addressed. Ready to run Phase 4 (Synthesis).")
    elif project.current_phase == 4:
        lines.append("Review chapter summaries. Ready to run Phase 5 (Draft Generation) with `--generate-drafts`.")
    elif project.current_phase == 5:
        lines.append("Review drafts in the `drafts/` folder. Project complete!")

    return "\n".join(lines)


def get_phase_name(phase: int) -> str:
    """Get human-readable phase name."""
    names = {
        1: "Initial Research",
        2: "Gap Analysis",
        3: "Gap Filling",
        4: "Synthesis",
        5: "Draft Generation"
    }
    return names.get(phase, f"Phase {phase}")


def handle_dry_run(args, project: BookProject) -> int:
    """Preview workflow without executing."""
    print(f"\n{'='*60}")
    print("DRY RUN PREVIEW")
    print(f"{'='*60}\n")

    print(f"Project: {project.title}")
    print(f"Project ID: {project.project_id}")
    print()

    print("CHAPTERS TO PROCESS:")
    total_subjects = 0
    for chapter in project.chapters:
        print(f"  Chapter {chapter.chapter_number}: {chapter.title}")
        print(f"    Subjects: {len(chapter.subjects)}")
        for subj in chapter.subjects[:5]:
            print(f"      - {subj}")
        if len(chapter.subjects) > 5:
            print(f"      ... and {len(chapter.subjects) - 5} more")
        total_subjects += len(chapter.subjects)
    print()

    print(f"TOTAL: {len(project.chapters)} chapters, {total_subjects} subjects")
    print()

    print("WORKFLOW PHASES:")
    start_phase = args.phase or 1
    end_phase = args.stop_after_phase or 5

    for phase in range(start_phase, end_phase + 1):
        phase_names = {
            1: "Initial Research - Search library for each subject",
            2: "Gap Analysis - Identify and prioritize research gaps",
            3: "Gap Filling - Fill gaps via Tavily web search" + (" (AUTO)" if args.auto_fill_gaps else " (MANUAL)"),
            4: "Synthesis - Compile and summarize research",
            5: "Draft Generation - Generate chapter drafts" + (" (ENABLED)" if args.generate_drafts else " (DISABLED)")
        }
        print(f"  Phase {phase}: {phase_names[phase]}")
    print()

    if args.auto_fill_gaps:
        print(f"AUTO-FILL: Enabled with budget of {args.tavily_budget} credits")
    else:
        print("AUTO-FILL: Disabled (gaps.md will be generated for manual review)")
    print()

    print("OUTPUT DIRECTORY:")
    output_dir = Path(args.output_dir) if args.output_dir else PROJECTS_DIR / project.project_id
    print(f"  {output_dir}")
    print()

    print("To execute this workflow, remove --dry-run flag.")
    return 0


def handle_workflow(args) -> int:
    """Handle main workflow execution."""
    project = None

    # Determine project source
    if args.resume:
        # Resume existing project
        project_dir = PROJECTS_DIR / args.resume
        if not project_dir.exists():
            matches = list(PROJECTS_DIR.glob(f"*{args.resume}*"))
            if len(matches) == 1:
                project_dir = matches[0]
            else:
                print(f"Error: Project not found: {args.resume}")
                return 1

        try:
            project = BookProject.load(project_dir)
            logger.info(f"Resumed project: {project.project_id}")
        except Exception as e:
            print(f"Error loading project: {e}")
            return 1

    elif args.outline_file:
        # Parse outline file
        outline_path = Path(args.outline_file)
        if not outline_path.exists():
            print(f"Error: File not found: {outline_path}")
            return 1

        try:
            project = parse_outline_file(outline_path, args.title)
        except Exception as e:
            print(f"Error parsing outline: {e}")
            return 1

    elif args.subjects_file:
        # Parse subjects file
        subjects_path = Path(args.subjects_file)
        if not subjects_path.exists():
            print(f"Error: File not found: {subjects_path}")
            return 1

        try:
            project = parse_subjects_file(subjects_path, args.title)
        except Exception as e:
            print(f"Error parsing subjects file: {e}")
            return 1

    elif args.subjects:
        # Create from command-line subjects
        try:
            project = parse_subjects_list(args.subjects, args.title)
        except Exception as e:
            print(f"Error creating project: {e}")
            return 1

    else:
        print("Error: No input specified. Provide an outline file, --subjects-file, or --subjects")
        return 1

    # Validate project
    warnings = validate_project(project)
    if warnings:
        for warning in warnings:
            logger.warning(warning)
        if not args.quiet:
            print(f"Warning: {len(warnings)} validation issues found")

    # Set project directory
    project_dir = Path(args.output_dir) if args.output_dir else PROJECTS_DIR / project.project_id
    project.project_dir = str(project_dir)
    project_dir.mkdir(parents=True, exist_ok=True)

    # Apply settings
    project.auto_fill_gaps = args.auto_fill_gaps
    project.tavily_credit_budget = args.tavily_budget

    # Dry run?
    if args.dry_run:
        return handle_dry_run(args, project)

    # Configure research options
    research_config = {
        'max_iterations': args.max_iterations,
        'use_graphrag': not args.no_graphrag,
        'use_rerank': not args.no_rerank,
        'generate_drafts': args.generate_drafts,
        'draft_style': args.draft_style,
    }

    # Handle gap approval
    # Priority order: approve-all-gaps > approve-priority > approve-gaps > gaps-file
    if args.approve_all_gaps:
        # Approve all gaps automatically
        approved_count = 0
        for gap in project.all_gaps:
            gap['approved'] = True
            approved_count += 1
        logger.info(f"Auto-approved all {approved_count} gaps")

    elif args.approve_priority:
        # Approve gaps by priority level
        priority_levels = {'high': ['high'], 'medium': ['high', 'medium'],
                          'low': ['high', 'medium', 'low'], 'all': ['high', 'medium', 'low']}
        approved_priorities = priority_levels.get(args.approve_priority, [])
        approved_count = 0
        for gap in project.all_gaps:
            if gap.get('priority', 'medium') in approved_priorities:
                gap['approved'] = True
                approved_count += 1
        logger.info(f"Auto-approved {approved_count} gaps with priority {args.approve_priority} or higher")

    elif args.approve_gaps:
        # Approve specific gaps by index
        try:
            indices = [int(i.strip()) - 1 for i in args.approve_gaps.split(',')]  # 1-indexed input
            approved_count = 0
            for i, gap in enumerate(project.all_gaps):
                if i in indices:
                    gap['approved'] = True
                    approved_count += 1
            logger.info(f"Auto-approved {approved_count} specific gaps")
        except ValueError:
            logger.warning(f"Invalid gap indices: {args.approve_gaps}")

    elif args.gaps_file:
        # Load from gaps.md file
        gaps_path = Path(args.gaps_file)
        if gaps_path.exists():
            try:
                with open(gaps_path, 'r') as f:
                    approved_queries = parse_gaps_markdown(f.read())
                logger.info(f"Loaded {len(approved_queries)} approved gaps from {gaps_path}")
                # Mark approved gaps in project
                for gap in project.all_gaps:
                    if gap.get('suggested_query') in approved_queries:
                        gap['approved'] = True
            except Exception as e:
                logger.warning(f"Error parsing gaps file: {e}")

    # Determine phases to run
    start_phase = args.phase or project.current_phase or 1
    end_phase = args.stop_after_phase or (args.phase if args.phase else 5)

    if not args.quiet:
        print(f"\n{'='*60}")
        print(f"RESEARCH WORKFLOW: {project.title}")
        print(f"{'='*60}")
        print(f"Project ID: {project.project_id}")
        print(f"Chapters: {len(project.chapters)}")
        print(f"Subjects: {project.total_subjects}")
        print(f"Phases: {start_phase} to {end_phase}")
        print(f"Output: {project_dir}")
        print(f"{'='*60}\n")

    # Save initial state
    project.save(project_dir)

    # Run workflow
    try:
        result = run_workflow(
            project=project,
            start_phase=start_phase,
            end_phase=end_phase,
            research_config=research_config,
            verbose=args.verbose,
        )

        # Write outputs
        outputs = write_all_outputs(project, project_dir)

        if args.format == 'json':
            print(generate_status_json(project))
        else:
            print(f"\n{'='*60}")
            print("WORKFLOW COMPLETE")
            print(f"{'='*60}\n")
            print(f"Project: {project.project_id}")
            print(f"Status: {project.status}")
            print(f"Final Phase: {project.current_phase}")
            print()
            print("OUTPUTS:")
            for name, path in outputs.items():
                if isinstance(path, list):
                    for p in path:
                        print(f"  - {p}")
                else:
                    print(f"  - {path}")
            print()

            # Show next steps
            if project.current_phase == 2 and not args.auto_fill_gaps:
                print("NEXT STEPS:")
                print(f"  1. Review gaps.md: {outputs.get('gaps')}")
                print(f"  2. Mark gaps to research with [x]")
                print(f"  3. Resume: python research_workflow.py --resume {project.project_id} --phase 3")
            elif project.current_phase < 5:
                print("NEXT STEPS:")
                print(f"  Resume: python research_workflow.py --resume {project.project_id} --phase {project.current_phase + 1}")

        return 0

    except KeyboardInterrupt:
        print("\nWorkflow interrupted. Saving checkpoint...")
        project.save(project_dir)
        print(f"Resume with: python research_workflow.py --resume {project.project_id}")
        return 130

    except Exception as e:
        logger.exception("Workflow error")
        project.save(project_dir)
        if args.format == 'json':
            print(json.dumps({'error': str(e), 'project_id': project.project_id}))
        else:
            print(f"\nError: {e}")
            print(f"Checkpoint saved. Resume with: python research_workflow.py --resume {project.project_id}")
        return 1


# =============================================================================
# MAIN ENTRY POINT
# =============================================================================

def main() -> int:
    """Main entry point."""
    parser = create_parser()
    args = parser.parse_args()

    # Set logging level
    if args.verbose:
        logging.getLogger().setLevel(logging.DEBUG)
    elif args.quiet:
        logging.getLogger().setLevel(logging.ERROR)

    # Ensure projects directory exists
    PROJECTS_DIR.mkdir(parents=True, exist_ok=True)

    # Handle special commands
    if args.list_projects:
        return handle_list_projects(args)

    if args.status:
        return handle_status(args)

    if args.context:
        return handle_context(args)

    # Main workflow
    return handle_workflow(args)


if __name__ == '__main__':
    sys.exit(main())
