#!/usr/bin/env python3
"""
Quarantine Manager - Review Documents Before Removal

Provides a soft-delete workflow for documents flagged for rejection:
1. QUARANTINE - Document is hidden from search but preserved for review
2. REVIEW - Human reviews the quarantine decision
3. RESTORE - Return document to active library
4. PURGE - Permanently delete (with confirmation)

The quarantine system preserves:
- Full document metadata
- All chunks and embeddings
- Curation history
- Rejection reasoning

Usage:
    # Quarantine a document (soft-delete)
    python quarantine_manager.py --quarantine DOC_001 --reason "Low curation score"

    # List quarantined documents
    python quarantine_manager.py --list

    # Review a quarantined document
    python quarantine_manager.py --review DOC_001

    # Restore a document to active library
    python quarantine_manager.py --restore DOC_001

    # Permanently purge (requires confirmation)
    python quarantine_manager.py --purge DOC_001 --confirm

    # Bulk operations
    python quarantine_manager.py --purge-all --older-than 30 --confirm
"""

import os
import sys
import json
import argparse
import logging
from datetime import datetime, timedelta
from typing import Dict, List, Any, Optional, Tuple
from pathlib import Path
from dataclasses import dataclass, asdict, field

# Add parent directory to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))
os.chdir(str(Path(__file__).parent.parent))

from pipeline.db_utils import get_db_connection, execute_query

logging.basicConfig(level=logging.INFO, format='%(levelname)s: %(message)s')
logger = logging.getLogger(__name__)


# =============================================================================
# CURATION STATUS VALUES
# =============================================================================
# These extend the existing quality_status enum concept

CURATION_STATUS = {
    'pending': 'Not yet curated',
    'approved': 'Approved for library',
    'quarantined': 'Flagged for review/removal',
    'restored': 'Restored from quarantine',
    'purged': 'Marked for permanent deletion'
}


# =============================================================================
# DATA CLASSES
# =============================================================================

@dataclass
class QuarantineRecord:
    """Record of a quarantine action."""
    document_id: str
    title: str
    quarantined_at: str
    quarantined_by: str  # 'curator', 'manual', 'auto'
    reason: str
    curation_score: Optional[float] = None
    curation_persona: Optional[str] = None
    review_notes: str = ""
    days_in_quarantine: int = 0


@dataclass
class QuarantineStats:
    """Statistics about quarantine queue."""
    total_quarantined: int = 0
    pending_review: int = 0
    by_reason: Dict[str, int] = field(default_factory=dict)
    oldest_days: int = 0
    avg_days: float = 0.0


# =============================================================================
# DATABASE OPERATIONS
# =============================================================================

def ensure_curation_columns():
    """Ensure the database has curation tracking columns."""
    with get_db_connection() as conn:
        with conn.cursor() as cur:
            # Check if columns exist
            cur.execute("""
                SELECT column_name
                FROM information_schema.columns
                WHERE table_name = 'documents'
                AND column_name IN ('curation_status', 'quarantined_at', 'quarantine_reason', 'curation_score')
            """)
            existing = {row[0] for row in cur.fetchall()}

            # Add missing columns
            if 'curation_status' not in existing:
                cur.execute("""
                    ALTER TABLE documents
                    ADD COLUMN IF NOT EXISTS curation_status VARCHAR(20) DEFAULT 'pending'
                """)
                logger.info("Added curation_status column")

            if 'quarantined_at' not in existing:
                cur.execute("""
                    ALTER TABLE documents
                    ADD COLUMN IF NOT EXISTS quarantined_at TIMESTAMP
                """)
                logger.info("Added quarantined_at column")

            if 'quarantine_reason' not in existing:
                cur.execute("""
                    ALTER TABLE documents
                    ADD COLUMN IF NOT EXISTS quarantine_reason TEXT
                """)
                logger.info("Added quarantine_reason column")

            if 'curation_score' not in existing:
                cur.execute("""
                    ALTER TABLE documents
                    ADD COLUMN IF NOT EXISTS curation_score NUMERIC(4,2)
                """)
                logger.info("Added curation_score column")

            conn.commit()


def quarantine_document(
    document_id: str,
    reason: str,
    quarantined_by: str = 'manual',
    curation_score: Optional[float] = None
) -> bool:
    """
    Move a document to quarantine status.

    The document remains in the database but is excluded from searches.
    """
    ensure_curation_columns()

    with get_db_connection() as conn:
        with conn.cursor() as cur:
            # Check document exists
            cur.execute("""
                SELECT document_id, title, curation_status
                FROM documents
                WHERE document_id = %s
            """, (document_id,))
            row = cur.fetchone()

            if not row:
                logger.error(f"Document {document_id} not found")
                return False

            current_status = row[2]
            if current_status == 'quarantined':
                logger.warning(f"Document {document_id} is already quarantined")
                return True

            # Update to quarantined status
            cur.execute("""
                UPDATE documents
                SET curation_status = 'quarantined',
                    quarantined_at = NOW(),
                    quarantine_reason = %s,
                    curation_score = %s,
                    needs_review = true,
                    updated_at = NOW()
                WHERE document_id = %s
            """, (f"[{quarantined_by}] {reason}", curation_score, document_id))

            conn.commit()
            logger.info(f"Quarantined document: {document_id}")
            return True


def restore_document(document_id: str, notes: str = "") -> bool:
    """
    Restore a quarantined document to active status.
    """
    with get_db_connection() as conn:
        with conn.cursor() as cur:
            # Check document is quarantined
            cur.execute("""
                SELECT curation_status FROM documents WHERE document_id = %s
            """, (document_id,))
            row = cur.fetchone()

            if not row:
                logger.error(f"Document {document_id} not found")
                return False

            if row[0] != 'quarantined':
                logger.warning(f"Document {document_id} is not quarantined (status: {row[0]})")
                return False

            # Restore document
            restore_note = f"\n[RESTORED {datetime.now().isoformat()}] {notes}" if notes else ""

            cur.execute("""
                UPDATE documents
                SET curation_status = 'restored',
                    quarantined_at = NULL,
                    needs_review = false,
                    notes = COALESCE(notes, '') || %s,
                    updated_at = NOW()
                WHERE document_id = %s
            """, (restore_note, document_id))

            conn.commit()
            logger.info(f"Restored document: {document_id}")
            return True


def purge_document(document_id: str, confirm: bool = False) -> Dict[str, Any]:
    """
    Permanently delete a quarantined document.

    This removes:
    - Document record
    - All chunks
    - All embeddings
    - File associations
    - Concept/topic links
    """
    result = {
        'document_id': document_id,
        'purged': False,
        'chunks_deleted': 0,
        'files_deleted': 0,
        'error': None
    }

    if not confirm:
        result['error'] = "Purge requires --confirm flag"
        return result

    with get_db_connection() as conn:
        with conn.cursor() as cur:
            # Verify document is quarantined
            cur.execute("""
                SELECT title, curation_status FROM documents WHERE document_id = %s
            """, (document_id,))
            row = cur.fetchone()

            if not row:
                result['error'] = f"Document {document_id} not found"
                return result

            title, status = row
            if status != 'quarantined':
                result['error'] = f"Document is not quarantined (status: {status}). Quarantine before purging."
                return result

            result['title'] = title

            # Delete in order to respect foreign keys

            # 1. Delete chunk_concepts
            cur.execute("""
                DELETE FROM chunk_concepts
                WHERE chunk_id IN (SELECT chunk_id FROM chunks WHERE document_id = %s)
            """, (document_id,))

            # 2. Delete document_concepts
            cur.execute("""
                DELETE FROM document_concepts WHERE document_id = %s
            """, (document_id,))

            # 3. Delete document_topics
            cur.execute("""
                DELETE FROM document_topics WHERE document_id = %s
            """, (document_id,))

            # 4. Delete chunks
            cur.execute("""
                DELETE FROM chunks WHERE document_id = %s
            """, (document_id,))
            result['chunks_deleted'] = cur.rowcount

            # 5. Delete files
            cur.execute("""
                DELETE FROM files WHERE document_id = %s
            """, (document_id,))
            result['files_deleted'] = cur.rowcount

            # 6. Delete document
            cur.execute("""
                DELETE FROM documents WHERE document_id = %s
            """, (document_id,))

            conn.commit()
            result['purged'] = True
            logger.info(f"Purged document: {document_id} ({result['chunks_deleted']} chunks)")

    return result


def get_quarantined_documents(
    limit: int = 100,
    older_than_days: Optional[int] = None
) -> List[QuarantineRecord]:
    """Get list of quarantined documents."""
    ensure_curation_columns()

    records = []

    with get_db_connection() as conn:
        with conn.cursor() as cur:
            query = """
                SELECT document_id, title, quarantined_at, quarantine_reason,
                       curation_score, notes
                FROM documents
                WHERE curation_status = 'quarantined'
            """
            params = []

            if older_than_days:
                query += " AND quarantined_at < NOW() - INTERVAL '%s days'"
                params.append(older_than_days)

            query += " ORDER BY quarantined_at DESC LIMIT %s"
            params.append(limit)

            cur.execute(query, params)

            for row in cur.fetchall():
                doc_id, title, quarantined_at, reason, score, notes = row

                # Parse quarantined_by from reason
                quarantined_by = 'manual'
                if reason and reason.startswith('['):
                    end = reason.find(']')
                    if end > 0:
                        quarantined_by = reason[1:end]
                        reason = reason[end+1:].strip()

                # Calculate days in quarantine
                days = 0
                if quarantined_at:
                    days = (datetime.now() - quarantined_at).days

                # Extract curation persona from notes if present
                persona = None
                if notes and 'persona' in notes.lower():
                    import re
                    match = re.search(r'"persona":\s*"([^"]+)"', notes)
                    if match:
                        persona = match.group(1)

                records.append(QuarantineRecord(
                    document_id=doc_id,
                    title=title or 'Untitled',
                    quarantined_at=quarantined_at.isoformat() if quarantined_at else '',
                    quarantined_by=quarantined_by,
                    reason=reason or '',
                    curation_score=float(score) if score else None,
                    curation_persona=persona,
                    days_in_quarantine=days
                ))

    return records


def get_quarantine_stats() -> QuarantineStats:
    """Get statistics about the quarantine queue."""
    ensure_curation_columns()

    stats = QuarantineStats()

    with get_db_connection() as conn:
        with conn.cursor() as cur:
            # Total quarantined
            cur.execute("""
                SELECT COUNT(*) FROM documents WHERE curation_status = 'quarantined'
            """)
            stats.total_quarantined = cur.fetchone()[0]

            # Pending review (quarantined + needs_review)
            cur.execute("""
                SELECT COUNT(*) FROM documents
                WHERE curation_status = 'quarantined' AND needs_review = true
            """)
            stats.pending_review = cur.fetchone()[0]

            # By reason category
            cur.execute("""
                SELECT
                    CASE
                        WHEN quarantine_reason LIKE '%%curator%%' THEN 'curator_rejected'
                        WHEN quarantine_reason LIKE '%%auto%%' THEN 'auto_rejected'
                        WHEN quarantine_reason LIKE '%%quality%%' THEN 'quality_issue'
                        ELSE 'manual'
                    END as category,
                    COUNT(*)
                FROM documents
                WHERE curation_status = 'quarantined'
                GROUP BY category
            """)
            stats.by_reason = dict(cur.fetchall())

            # Age statistics
            cur.execute("""
                SELECT
                    MAX(EXTRACT(DAY FROM NOW() - quarantined_at)) as oldest,
                    AVG(EXTRACT(DAY FROM NOW() - quarantined_at)) as avg_days
                FROM documents
                WHERE curation_status = 'quarantined' AND quarantined_at IS NOT NULL
            """)
            row = cur.fetchone()
            if row and row[0]:
                stats.oldest_days = int(row[0])
                stats.avg_days = round(float(row[1]), 1) if row[1] else 0

    return stats


def review_document(document_id: str) -> Optional[Dict[str, Any]]:
    """Get detailed review information for a quarantined document."""
    with get_db_connection() as conn:
        with conn.cursor() as cur:
            cur.execute("""
                SELECT d.document_id, d.title, d.author_id, a.name as author_name,
                       d.publication_year, d.primary_category, d.quality_score,
                       d.quality_status, d.curation_status, d.curation_score,
                       d.quarantined_at, d.quarantine_reason, d.notes,
                       d.word_count, d.created_at,
                       (SELECT COUNT(*) FROM chunks WHERE document_id = d.document_id) as chunk_count
                FROM documents d
                LEFT JOIN authors a ON d.author_id = a.author_id
                WHERE d.document_id = %s
            """, (document_id,))

            row = cur.fetchone()
            if not row:
                return None

            # Get sample chunk for content preview
            cur.execute("""
                SELECT chunk_text FROM chunks
                WHERE document_id = %s
                ORDER BY chunk_sequence LIMIT 1
            """, (document_id,))
            chunk_row = cur.fetchone()
            sample_text = chunk_row[0][:500] + "..." if chunk_row else "No content available"

            return {
                'document_id': row[0],
                'title': row[1],
                'author_id': row[2],
                'author_name': row[3],
                'publication_year': row[4],
                'primary_category': row[5],
                'quality_score': row[6],
                'quality_status': row[7],
                'curation_status': row[8],
                'curation_score': float(row[9]) if row[9] else None,
                'quarantined_at': row[10].isoformat() if row[10] else None,
                'quarantine_reason': row[11],
                'notes': row[12],
                'word_count': row[13],
                'created_at': row[14].isoformat() if row[14] else None,
                'chunk_count': row[15],
                'sample_text': sample_text
            }


def bulk_quarantine_by_score(
    max_score: float,
    persona: Optional[str] = None,
    dry_run: bool = True
) -> Dict[str, Any]:
    """Quarantine all documents below a curation score threshold."""
    ensure_curation_columns()

    result = {
        'threshold': max_score,
        'documents_found': 0,
        'documents_quarantined': 0,
        'dry_run': dry_run,
        'documents': []
    }

    with get_db_connection() as conn:
        with conn.cursor() as cur:
            # Find documents below threshold
            query = """
                SELECT document_id, title, curation_score
                FROM documents
                WHERE curation_score IS NOT NULL
                  AND curation_score < %s
                  AND curation_status != 'quarantined'
            """
            params = [max_score]

            cur.execute(query, params)
            documents = cur.fetchall()
            result['documents_found'] = len(documents)

            for doc_id, title, score in documents:
                result['documents'].append({
                    'document_id': doc_id,
                    'title': title,
                    'score': float(score) if score else None
                })

                if not dry_run:
                    quarantine_document(
                        doc_id,
                        reason=f"Score {score:.1f} below threshold {max_score}",
                        quarantined_by='auto',
                        curation_score=float(score) if score else None
                    )
                    result['documents_quarantined'] += 1

    return result


def bulk_purge(
    older_than_days: int,
    confirm: bool = False
) -> Dict[str, Any]:
    """Purge all quarantined documents older than specified days."""
    result = {
        'older_than_days': older_than_days,
        'documents_found': 0,
        'documents_purged': 0,
        'confirm': confirm,
        'documents': []
    }

    if not confirm:
        result['error'] = "Bulk purge requires --confirm flag"

    records = get_quarantined_documents(limit=1000, older_than_days=older_than_days)
    result['documents_found'] = len(records)

    for record in records:
        result['documents'].append({
            'document_id': record.document_id,
            'title': record.title,
            'days_in_quarantine': record.days_in_quarantine
        })

        if confirm:
            purge_result = purge_document(record.document_id, confirm=True)
            if purge_result['purged']:
                result['documents_purged'] += 1

    return result


# =============================================================================
# OUTPUT FORMATTERS
# =============================================================================

def print_quarantine_list(records: List[QuarantineRecord], format: str = 'text') -> None:
    """Print list of quarantined documents."""
    if format == 'json':
        print(json.dumps([asdict(r) for r in records], indent=2))
        return

    print("\n" + "=" * 80)
    print("QUARANTINED DOCUMENTS")
    print("=" * 80)
    print(f"Total: {len(records)}\n")

    if not records:
        print("No quarantined documents.")
        return

    print(f"{'Document ID':<45} {'Score':<7} {'Days':<6} {'Reason'}")
    print("-" * 80)

    for r in records:
        score_str = f"{r.curation_score:.1f}" if r.curation_score else "N/A"
        reason_short = r.reason[:25] + "..." if len(r.reason) > 25 else r.reason
        print(f"{r.document_id:<45} {score_str:<7} {r.days_in_quarantine:<6} {reason_short}")

    print("\nCommands:")
    print("  Review:  python quarantine_manager.py --review DOC_ID")
    print("  Restore: python quarantine_manager.py --restore DOC_ID")
    print("  Purge:   python quarantine_manager.py --purge DOC_ID --confirm")


def print_review(doc_info: Dict[str, Any], format: str = 'text') -> None:
    """Print detailed review of a quarantined document."""
    if format == 'json':
        print(json.dumps(doc_info, indent=2, default=str))
        return

    print("\n" + "=" * 80)
    print("QUARANTINE REVIEW")
    print("=" * 80)

    print(f"\nDocument ID: {doc_info['document_id']}")
    print(f"Title: {doc_info['title']}")
    print(f"Author: {doc_info['author_name'] or 'Unknown'}")
    print(f"Year: {doc_info['publication_year'] or 'Unknown'}")
    print(f"Category: {doc_info['primary_category'] or 'Unclassified'}")
    print()

    print("STATUS")
    print("-" * 40)
    print(f"Curation Status: {doc_info['curation_status']}")
    print(f"Curation Score: {doc_info['curation_score'] or 'N/A'}")
    print(f"Quality Score: {doc_info['quality_score'] or 'N/A'}")
    print(f"Quality Status: {doc_info['quality_status'] or 'N/A'}")
    print()

    print("QUARANTINE INFO")
    print("-" * 40)
    print(f"Quarantined At: {doc_info['quarantined_at'] or 'N/A'}")
    print(f"Reason: {doc_info['quarantine_reason'] or 'No reason provided'}")
    print()

    print("DOCUMENT INFO")
    print("-" * 40)
    print(f"Word Count: {doc_info['word_count'] or 'Unknown'}")
    print(f"Chunk Count: {doc_info['chunk_count']}")
    print(f"Created: {doc_info['created_at']}")
    print()

    print("CONTENT PREVIEW")
    print("-" * 40)
    print(doc_info['sample_text'])
    print()

    print("ACTIONS")
    print("-" * 40)
    print(f"  Restore: python quarantine_manager.py --restore {doc_info['document_id']}")
    print(f"  Purge:   python quarantine_manager.py --purge {doc_info['document_id']} --confirm")


def print_stats(stats: QuarantineStats, format: str = 'text') -> None:
    """Print quarantine statistics."""
    if format == 'json':
        print(json.dumps(asdict(stats), indent=2))
        return

    print("\n" + "=" * 50)
    print("QUARANTINE STATISTICS")
    print("=" * 50)
    print(f"Total Quarantined: {stats.total_quarantined}")
    print(f"Pending Review: {stats.pending_review}")
    print(f"Oldest: {stats.oldest_days} days")
    print(f"Average Age: {stats.avg_days} days")
    print()
    print("By Reason:")
    for reason, count in stats.by_reason.items():
        print(f"  {reason}: {count}")


# =============================================================================
# CLI INTERFACE
# =============================================================================

def main():
    parser = argparse.ArgumentParser(
        description="Quarantine Manager - Review documents before removal",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  python quarantine_manager.py --list
  python quarantine_manager.py --quarantine DOC_001 --reason "Low quality"
  python quarantine_manager.py --review DOC_001
  python quarantine_manager.py --restore DOC_001
  python quarantine_manager.py --purge DOC_001 --confirm
  python quarantine_manager.py --stats
  python quarantine_manager.py --bulk-quarantine --max-score 4.0 --dry-run
  python quarantine_manager.py --purge-all --older-than 30 --confirm
        """
    )

    # Operations
    ops = parser.add_argument_group('Operations')
    ops.add_argument('--list', '-l', action='store_true',
                    help='List quarantined documents')
    ops.add_argument('--quarantine', '-q', metavar='DOC_ID',
                    help='Quarantine a document')
    ops.add_argument('--review', '-r', metavar='DOC_ID',
                    help='Review a quarantined document')
    ops.add_argument('--restore', metavar='DOC_ID',
                    help='Restore a document from quarantine')
    ops.add_argument('--purge', metavar='DOC_ID',
                    help='Permanently delete a quarantined document')
    ops.add_argument('--stats', action='store_true',
                    help='Show quarantine statistics')

    # Bulk operations
    bulk = parser.add_argument_group('Bulk Operations')
    bulk.add_argument('--bulk-quarantine', action='store_true',
                     help='Quarantine documents below score threshold')
    bulk.add_argument('--purge-all', action='store_true',
                     help='Purge all quarantined documents older than --older-than days')

    # Options
    opts = parser.add_argument_group('Options')
    opts.add_argument('--reason', default='Manual quarantine',
                     help='Reason for quarantine')
    opts.add_argument('--max-score', type=float, default=4.0,
                     help='Score threshold for bulk quarantine (default: 4.0)')
    opts.add_argument('--older-than', type=int, metavar='DAYS',
                     help='Filter by days in quarantine')
    opts.add_argument('--limit', type=int, default=100,
                     help='Maximum documents to list (default: 100)')
    opts.add_argument('--confirm', action='store_true',
                     help='Confirm destructive operations')
    opts.add_argument('--dry-run', action='store_true',
                     help='Preview without making changes')
    opts.add_argument('--format', '-f', choices=['text', 'json'], default='text',
                     help='Output format')
    opts.add_argument('--notes', default='',
                     help='Notes for restore operation')

    args = parser.parse_args()

    try:
        # List quarantined documents
        if args.list:
            records = get_quarantined_documents(
                limit=args.limit,
                older_than_days=args.older_than
            )
            print_quarantine_list(records, args.format)
            return

        # Show stats
        if args.stats:
            stats = get_quarantine_stats()
            print_stats(stats, args.format)
            return

        # Quarantine a document
        if args.quarantine:
            success = quarantine_document(
                args.quarantine,
                reason=args.reason,
                quarantined_by='manual'
            )
            if success:
                print(f"Document {args.quarantine} quarantined.")
            else:
                print("Quarantine failed.")
                sys.exit(1)
            return

        # Review a document
        if args.review:
            doc_info = review_document(args.review)
            if doc_info:
                print_review(doc_info, args.format)
            else:
                print(f"Document {args.review} not found.")
                sys.exit(1)
            return

        # Restore a document
        if args.restore:
            success = restore_document(args.restore, notes=args.notes)
            if success:
                print(f"Document {args.restore} restored to active library.")
            else:
                print("Restore failed.")
                sys.exit(1)
            return

        # Purge a document
        if args.purge:
            if not args.confirm:
                print("WARNING: This will permanently delete the document and all its data.")
                print(f"To confirm, run: python quarantine_manager.py --purge {args.purge} --confirm")
                sys.exit(1)

            result = purge_document(args.purge, confirm=True)
            if args.format == 'json':
                print(json.dumps(result, indent=2))
            elif result['purged']:
                print(f"Document {args.purge} permanently deleted.")
                print(f"  Chunks deleted: {result['chunks_deleted']}")
            else:
                print(f"Purge failed: {result.get('error', 'Unknown error')}")
                sys.exit(1)
            return

        # Bulk quarantine by score
        if args.bulk_quarantine:
            result = bulk_quarantine_by_score(
                max_score=args.max_score,
                dry_run=args.dry_run
            )
            if args.format == 'json':
                print(json.dumps(result, indent=2))
            else:
                action = "Would quarantine" if args.dry_run else "Quarantined"
                print(f"\n{action} {result['documents_found']} documents with score < {args.max_score}")
                for doc in result['documents'][:10]:
                    print(f"  {doc['document_id']}: {doc['score']:.1f}")
                if len(result['documents']) > 10:
                    print(f"  ... and {len(result['documents']) - 10} more")
            return

        # Bulk purge
        if args.purge_all:
            if not args.older_than:
                print("Error: --purge-all requires --older-than DAYS")
                sys.exit(1)

            result = bulk_purge(
                older_than_days=args.older_than,
                confirm=args.confirm
            )
            if args.format == 'json':
                print(json.dumps(result, indent=2))
            else:
                if not args.confirm:
                    print(f"\nWould purge {result['documents_found']} documents older than {args.older_than} days:")
                    for doc in result['documents'][:10]:
                        print(f"  {doc['document_id']} ({doc['days_in_quarantine']} days)")
                    print("\nTo confirm: add --confirm flag")
                else:
                    print(f"\nPurged {result['documents_purged']} documents.")
            return

        # Default: show help
        parser.print_help()

    except KeyboardInterrupt:
        print("\nCancelled.")
        sys.exit(1)
    except Exception as e:
        logger.error(f"Error: {e}")
        if args.format == 'json':
            print(json.dumps({'error': str(e)}))
        sys.exit(1)


if __name__ == '__main__':
    main()
