#!/usr/bin/env python3
"""
RDF Outline - Generate book outline from natural language prompt

Creates a structured outline.yaml file from a topic description.

Usage:
    rdf outline "Topic description" --chapters 5 [--pages 100] [--style academic]
    rdf outline "Topic" --chapters 5 --preview  # Preview without saving
"""

import argparse
import json
import sys
import yaml
import os
from pathlib import Path
from datetime import datetime
import hashlib

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))

from pipeline.cli_utils import success_response, error_response, ErrorCodes


def generate_outline_id() -> str:
    """Generate a unique outline/book ID."""
    timestamp = datetime.now().strftime("%Y%m%d%H%M%S")
    hash_part = hashlib.md5(str(datetime.now().timestamp()).encode()).hexdigest()[:6]
    return f"BOOK_{timestamp}_{hash_part}"


def generate_outline(
    topic: str,
    chapters: int,
    pages: int = None,
    style: str = "academic"
) -> dict:
    """
    Generate an outline structure.

    In a full implementation, this would use Claude to generate
    intelligent chapter breakdowns. For now, creates a template structure.
    """
    pages_per_chapter = (pages // chapters) if pages else 20

    outline = {
        "title": topic,
        "subtitle": None,
        "target_pages": pages or (chapters * 20),
        "style": style,
        "research": {
            "strict_library_only": True,
            "min_coverage_threshold": 50
        },
        "chapters": []
    }

    # Generate chapter placeholders
    for i in range(1, chapters + 1):
        chapter = {
            "number": i,
            "title": f"Chapter {i}",
            "target_pages": pages_per_chapter,
            "synopsis": f"Chapter {i} content for: {topic}",
            "key_topics": [],
            "research_queries": []
        }
        outline["chapters"].append(chapter)

    return outline


def save_outline(outline: dict, book_id: str) -> Path:
    """Save outline to book_projects directory."""
    project_dir = Path("book_projects") / book_id
    project_dir.mkdir(parents=True, exist_ok=True)

    outline_path = project_dir / "outline.yaml"

    with open(outline_path, 'w', encoding='utf-8') as f:
        yaml.dump(outline, f, default_flow_style=False, allow_unicode=True, sort_keys=False)

    # Also create initial workflow_state.json
    state = {
        "book_id": book_id,
        "created_at": datetime.now().isoformat(),
        "current_phase": 0,
        "status": "initialized",
        "outline_path": str(outline_path)
    }

    state_path = project_dir / "workflow_state.json"
    with open(state_path, 'w') as f:
        json.dump(state, f, indent=2)

    return outline_path


def main():
    parser = argparse.ArgumentParser(
        description="Generate book outline from topic",
        formatter_class=argparse.RawDescriptionHelpFormatter
    )
    parser.add_argument("topic", help="Topic or title for the book")
    parser.add_argument("--chapters", type=int, required=True,
                        help="Number of chapters")
    parser.add_argument("--pages", type=int, default=None,
                        help="Target page count")
    parser.add_argument("--style", default="academic",
                        choices=["academic", "accessible", "narrative", "popular"],
                        help="Writing style")
    parser.add_argument("--preview", action="store_true",
                        help="Preview outline without saving")
    parser.add_argument("--format", choices=["json", "yaml"], default="json",
                        help="Output format")

    args = parser.parse_args()

    try:
        # Generate outline
        outline = generate_outline(
            topic=args.topic,
            chapters=args.chapters,
            pages=args.pages,
            style=args.style
        )

        if args.preview:
            # Just show the outline
            if args.format == "json":
                response = success_response(
                    "Outline preview generated",
                    data={
                        "preview": True,
                        "outline": outline
                    }
                )
                response.print_json()
            else:
                print(yaml.dump(outline, default_flow_style=False, allow_unicode=True))
            return 0

        # Save outline
        book_id = generate_outline_id()
        outline_path = save_outline(outline, book_id)

        response = success_response(
            f"Outline generated for: {args.topic}",
            data={
                "outline_path": str(outline_path),
                "book_id": book_id,
                "proposed_structure": {
                    "title": outline["title"],
                    "chapters": len(outline["chapters"]),
                    "estimated_pages": outline["target_pages"]
                }
            }
        )
        response.data["next_suggested_commands"] = [
            f"rdf book {outline_path} --plan",
            f"rdf book {outline_path} --autonomy supervised"
        ]

        if args.format == "json":
            response.print_json()
        else:
            print(f"Created: {outline_path}")
            print(f"Book ID: {book_id}")
            print(f"\nNext steps:")
            print(f"  rdf book {outline_path} --plan")
            print(f"  rdf book {outline_path} --autonomy supervised")

        return 0

    except Exception as e:
        resp = error_response(
            ErrorCodes.UNKNOWN_ERROR,
            str(e)
        )
        resp.print_json()
        return 1


if __name__ == "__main__":
    sys.exit(main())
