#!/usr/bin/env python3
"""
RDF Fetch - Retrieve bounded chunk text

Returns chunk content with optional character limit for context management.

Usage:
    rdf fetch <chunk_id> [--max-chars N] [--format json]
    rdf fetch <result_id> --from-search [--max-chars N]
"""

import argparse
import json
import sys
from pathlib import Path

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))

from pipeline.cli_utils import success_response, error_response, ErrorCodes
from pipeline.db_utils import get_db_connection


def fetch_chunk(chunk_id: str, max_chars: int = None) -> dict:
    """Fetch a chunk by ID with optional character limit."""
    conn = get_db_connection()
    cursor = conn.cursor()

    try:
        # Try to find the chunk
        cursor.execute("""
            SELECT
                c.chunk_id,
                c.document_id,
                c.content,
                c.chunk_index,
                c.start_char,
                c.end_char,
                d.title,
                d.author
            FROM chunks c
            JOIN documents d ON c.document_id = d.document_id
            WHERE c.chunk_id = %s
        """, (chunk_id,))

        row = cursor.fetchone()
        if not row:
            return None

        content = row[2]
        if max_chars and len(content) > max_chars:
            content = content[:max_chars] + "..."
            truncated = True
        else:
            truncated = False

        return {
            "chunk_id": row[0],
            "document_id": row[1],
            "content": content,
            "chunk_index": row[3],
            "char_range": [row[4], row[5]],
            "source": {
                "title": row[6],
                "author": row[7]
            },
            "truncated": truncated,
            "original_length": len(row[2]),
            "returned_length": len(content)
        }
    finally:
        cursor.close()
        conn.close()


def fetch_by_search_result(result_id: str, max_chars: int = None) -> dict:
    """Fetch content from a search result ID."""
    # Search results typically encode chunk_id or document_id
    # Try as chunk_id first
    result = fetch_chunk(result_id, max_chars)
    if result:
        return result

    # Try as document_id - return first chunk
    conn = get_db_connection()
    cursor = conn.cursor()

    try:
        cursor.execute("""
            SELECT chunk_id FROM chunks
            WHERE document_id = %s
            ORDER BY chunk_index
            LIMIT 1
        """, (result_id,))

        row = cursor.fetchone()
        if row:
            return fetch_chunk(row[0], max_chars)
        return None
    finally:
        cursor.close()
        conn.close()


def main():
    parser = argparse.ArgumentParser(
        description="Retrieve bounded chunk text",
        formatter_class=argparse.RawDescriptionHelpFormatter
    )
    parser.add_argument("id", help="Chunk ID or result ID to fetch")
    parser.add_argument("--max-chars", type=int, default=None,
                        help="Maximum characters to return")
    parser.add_argument("--from-search", action="store_true",
                        help="Treat ID as search result ID")
    parser.add_argument("--format", choices=["json", "text"], default="json",
                        help="Output format")

    args = parser.parse_args()

    try:
        if args.from_search:
            result = fetch_by_search_result(args.id, args.max_chars)
        else:
            result = fetch_chunk(args.id, args.max_chars)

        if not result:
            resp = error_response(
                ErrorCodes.DOCUMENT_NOT_FOUND,
                f"Chunk not found: {args.id}",
                context={"query": args.id}
            )
            if args.format == "json":
                resp.print_json()
            else:
                print(f"Error: {resp.message}")
            return 1

        if args.format == "json":
            response = success_response(
                f"Retrieved chunk {args.id}",
                data=result
            )
            response.print_json()
        else:
            print(f"# {result['source']['title']}")
            print(f"# Chunk: {result['chunk_id']}")
            print(f"# Characters: {result['returned_length']}/{result['original_length']}")
            print()
            print(result['content'])

        return 0

    except Exception as e:
        resp = error_response(
            ErrorCodes.DATABASE_ERROR,
            str(e)
        )
        if args.format == "json":
            resp.print_json()
        else:
            print(f"Error: {e}")
        return 1


if __name__ == "__main__":
    sys.exit(main())
