#!/usr/bin/env python3
"""
RDF Essay - Single-command essay generation

Generates a complete essay on a topic using the library.

OPT-IN GENERATION MODEL:
    This tool follows the "Opt-in Generation" architecture:

    DEFAULT MODE (brief):
        Returns an essay brief with research and structure for Claude Code to write.
        Claude Code maintains voice consistency and authorship.

    DELEGATED MODE (--delegate-drafting):
        Uses the internal LLM to generate the essay draft.
        Use this for batch processing or when explicitly delegating to the CLI.

Usage:
    # Get essay brief for Claude Code to write (DEFAULT)
    rdf essay "Topic" --pages 20

    # Have CLI generate essay using internal model (OPT-IN)
    rdf essay "Topic" --pages 20 --delegate-drafting

    # With writing style
    rdf essay "Topic" --pages 15 --style accessible
"""

import argparse
import json
import sys
import logging
from pathlib import Path
from datetime import datetime
import hashlib
from typing import Dict, Any, Optional, List

# Add parent to path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))
sys.path.insert(0, str(Path(__file__).parent))

from pipeline.cli_utils import success_response, error_response, ErrorCodes

# Setup logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# Writing style presets (same as book)
WRITING_PRESETS = {
    'academic': """Write in formal academic style:
- Use precise, scholarly language
- Maintain objective tone
- Include appropriate hedging for uncertain claims
- Use topic sentences and clear paragraph structure
- Prefer active voice where appropriate""",

    'accessible': """Write in academic but accessible style:
- Explain technical terms when first used
- Use concrete examples to illustrate abstract concepts
- Vary sentence length for readability
- Avoid unnecessary jargon
- Maintain scholarly rigor while being approachable""",

    'narrative': """Write in engaging narrative style:
- Use storytelling techniques where appropriate
- Create flow between paragraphs
- Vary rhythm and pacing
- Make historical figures come alive
- Balance narrative with scholarly content""",

    'popular': """Write for general readers:
- Assume no prior knowledge of the subject
- Define all technical terms
- Use analogies and comparisons to familiar concepts
- Engage the reader's curiosity
- Break up dense content with examples""",
}


def generate_essay_id() -> str:
    """Generate a unique essay ID."""
    timestamp = datetime.now().strftime("%Y%m%d%H%M%S")
    hash_part = hashlib.md5(str(datetime.now().timestamp()).encode()).hexdigest()[:6]
    return f"ESSAY_{timestamp}_{hash_part}"


def create_essay_project(topic: str, pages: int, style: str) -> tuple:
    """Create essay project directory structure."""
    essay_id = generate_essay_id()
    project_dir = Path("essays") / essay_id
    project_dir.mkdir(parents=True, exist_ok=True)

    # Create initial state
    state = {
        "essay_id": essay_id,
        "topic": topic,
        "target_pages": pages,
        "style": style,
        "created_at": datetime.now().isoformat(),
        "current_phase": "initialized",
        "phases_completed": []
    }

    state_path = project_dir / "workflow_state.json"
    with open(state_path, 'w') as f:
        json.dump(state, f, indent=2)

    return essay_id, project_dir


def conduct_research(topic: str, limit: int = 20) -> Dict[str, Any]:
    """
    Conduct research on the topic using the library.

    Returns research results for Claude Code to use.
    """
    research_results = {
        'topic': topic,
        'sources': [],
        'synthesis': '',
        'key_quotes': [],
        'gaps': [],
    }

    try:
        from research_agent import ResearchAgent, format_markdown_report

        agent = ResearchAgent(
            max_iterations=5,
            use_graphrag=True,
            use_rerank=True,
            auto_fill_gaps=False,
        )

        # Run research
        result = agent.research(topic, max_results=limit)

        if result:
            research_results['synthesis'] = result.synthesis or ''
            research_results['sources'] = [
                {
                    'document_id': r.get('document_id', ''),
                    'title': r.get('title', 'Unknown'),
                    'author': r.get('author', 'Unknown'),
                    'chunk_text': r.get('chunk_text', '')[:500],
                    'relevance_score': r.get('relevance_score', 0),
                }
                for r in (result.results or [])[:20]
            ]
            research_results['key_quotes'] = [
                {
                    'text': r.get('chunk_text', '')[:300],
                    'source': r.get('document_id', ''),
                    'title': r.get('title', 'Unknown'),
                }
                for r in (result.results or [])[:10]
            ]
            research_results['gaps'] = [
                {'query': g.query, 'priority': g.priority}
                for g in (result.gaps or [])
            ]

    except ImportError:
        logger.warning("ResearchAgent not available, using basic search")
        # Fall back to basic search
        try:
            from db_utils import hybrid_search_with_rerank
            results = hybrid_search_with_rerank(topic, limit=limit)
            research_results['sources'] = [
                {
                    'document_id': r.get('document_id', ''),
                    'title': r.get('title', 'Unknown'),
                    'author': r.get('author', 'Unknown'),
                    'chunk_text': r.get('chunk_text', '')[:500],
                }
                for r in results
            ]
            research_results['synthesis'] = f"Found {len(results)} relevant sources on '{topic}'."
        except Exception as e:
            logger.warning(f"Basic search failed: {e}")
            research_results['synthesis'] = f"Research pending for topic: {topic}"

    except Exception as e:
        logger.error(f"Research failed: {e}")
        research_results['synthesis'] = f"Research failed: {str(e)}"

    return research_results


def generate_essay_brief(
    topic: str,
    pages: int,
    style: str,
    research: Dict[str, Any]
) -> Dict[str, Any]:
    """
    Generate an essay brief for Claude Code to write.

    This is the DEFAULT mode when --delegate-drafting is NOT specified.
    Returns research, structure, and guidance for Claude Code to write the essay.
    """
    word_count_target = pages * 250  # ~250 words per page

    # Suggested sections based on page count
    if pages <= 5:
        suggested_sections = ['Introduction', 'Main Body', 'Conclusion']
    elif pages <= 15:
        suggested_sections = [
            'Introduction',
            'Background/Context',
            'Main Argument 1',
            'Main Argument 2',
            'Analysis/Discussion',
            'Conclusion'
        ]
    else:
        suggested_sections = [
            'Introduction',
            'Background/Context',
            'Main Argument 1',
            'Main Argument 2',
            'Main Argument 3',
            'Counter-arguments/Discussion',
            'Synthesis',
            'Conclusion'
        ]

    return {
        'topic': topic,
        'target_pages': pages,
        'target_word_count': word_count_target,

        'writing_style': WRITING_PRESETS.get(style, WRITING_PRESETS['accessible']),
        'style_name': style,

        'research': research,

        'suggested_structure': {
            'sections': suggested_sections,
            'words_per_section': word_count_target // len(suggested_sections),
        },

        'instructions': (
            f"Claude Code should write a {pages}-page essay (~{word_count_target} words) on '{topic}'. "
            f"Use the research synthesis and quotes provided, citing sources with [DOC_ID] format. "
            f"Follow the {style} writing style guidelines. Structure the essay with clear sections: "
            f"{', '.join(suggested_sections)}. Mark any gaps with [RESEARCH NEEDED: topic] placeholders."
        ),

        'note': 'This is an essay brief. Claude Code should write the draft using this information.',
    }


def generate_essay_delegated(
    topic: str,
    pages: int,
    style: str,
    research: Dict[str, Any],
    project_dir: Path
) -> Dict[str, Any]:
    """
    Generate an essay using the internal LLM.

    This is DELEGATED mode when --delegate-drafting is specified.
    """
    word_count_target = pages * 250

    try:
        from research_agent import LLMInterface
        llm = LLMInterface()

        # Build prompt with research
        research_summary = research.get('synthesis', '')
        quotes_text = '\n'.join([
            f"- \"{q['text'][:200]}...\" — {q.get('title', 'Unknown')} [{q.get('source', '')}]"
            for q in research.get('key_quotes', [])[:10]
        ])

        prompt = f"""Write a {pages}-page essay (~{word_count_target} words) on: {topic}

RESEARCH SUMMARY:
{research_summary}

KEY QUOTES/EVIDENCE:
{quotes_text}

WRITING STYLE:
{WRITING_PRESETS.get(style, WRITING_PRESETS['accessible'])}

Requirements:
1. Include an introduction, body sections, and conclusion
2. Use evidence from the research to support claims
3. Cite sources using [DOC_ID] format
4. Mark any information gaps with [RESEARCH NEEDED: topic]
5. Target approximately {word_count_target} words

Write the complete essay:"""

        messages = [
            {"role": "system", "content": "You are an expert academic writer."},
            {"role": "user", "content": prompt}
        ]

        draft = llm.chat(messages, temperature=0.7)

        # Save draft
        draft_path = project_dir / "essay.md"
        with open(draft_path, 'w') as f:
            f.write(f"# {topic}\n\n{draft}")

        return {
            'success': True,
            'mode': 'delegated',
            'draft_path': str(draft_path),
            'word_count': len(draft.split()),
            'target_word_count': word_count_target,
        }

    except ImportError:
        return {
            'success': False,
            'mode': 'delegated',
            'error': 'LLM not available for delegated drafting',
        }
    except Exception as e:
        return {
            'success': False,
            'mode': 'delegated',
            'error': str(e),
        }


def run_essay_workflow(
    topic: str,
    pages: int,
    style: str,
    autonomy: str,
    evidence_first: bool,
    strict: bool,
    delegate_drafting: bool = False
) -> dict:
    """
    Run the essay generation workflow.

    If delegate_drafting is False (default): Returns essay brief for Claude Code
    If delegate_drafting is True: Uses internal LLM to generate draft
    """
    essay_id, project_dir = create_essay_project(topic, pages, style)

    # Conduct research
    logger.info(f"Researching topic: {topic}")
    research = conduct_research(topic, limit=30)

    # Save research
    research_path = project_dir / "research.json"
    with open(research_path, 'w') as f:
        json.dump(research, f, indent=2)

    # Determine mode
    if delegate_drafting:
        # DELEGATED MODE: Use internal LLM
        logger.info("DELEGATED MODE: Generating essay with internal LLM")
        draft_result = generate_essay_delegated(
            topic, pages, style, research, project_dir
        )

        # Update state
        state_path = project_dir / "workflow_state.json"
        with open(state_path, 'r') as f:
            state = json.load(f)
        state["current_phase"] = "draft_complete" if draft_result.get('success') else "draft_failed"
        state["phases_completed"] = ["research", "draft"] if draft_result.get('success') else ["research"]
        state["mode"] = "delegated"
        with open(state_path, 'w') as f:
            json.dump(state, f, indent=2)

        return {
            "essay_id": essay_id,
            "project_dir": str(project_dir),
            "topic": topic,
            "target_pages": pages,
            "style": style,
            "mode": "delegated",
            "draft_result": draft_result,
            "artifacts": {
                "research": str(research_path),
                "draft": str(project_dir / "essay.md") if draft_result.get('success') else None,
            }
        }
    else:
        # DEFAULT MODE: Return brief for Claude Code
        logger.info("DEFAULT MODE: Generating essay brief for Claude Code")
        brief = generate_essay_brief(topic, pages, style, research)

        # Save brief
        brief_path = project_dir / "essay_brief.json"
        with open(brief_path, 'w') as f:
            json.dump(brief, f, indent=2)

        # Update state
        state_path = project_dir / "workflow_state.json"
        with open(state_path, 'r') as f:
            state = json.load(f)
        state["current_phase"] = "brief_ready"
        state["phases_completed"] = ["research", "brief"]
        state["mode"] = "brief"
        with open(state_path, 'w') as f:
            json.dump(state, f, indent=2)

        return {
            "essay_id": essay_id,
            "project_dir": str(project_dir),
            "topic": topic,
            "target_pages": pages,
            "style": style,
            "mode": "brief",
            "brief": brief,
            "artifacts": {
                "research": str(research_path),
                "brief": str(brief_path),
            },
            "note": "Claude Code should write the essay draft using this brief"
        }


def main():
    parser = argparse.ArgumentParser(
        description="Single-command essay generation",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Get essay brief for Claude Code to write (DEFAULT)
  rdf essay "The History of Jazz" --pages 15

  # Have CLI generate essay using internal model (OPT-IN)
  rdf essay "The History of Jazz" --pages 15 --delegate-drafting

  # With specific writing style
  rdf essay "Climate Change" --pages 20 --style academic
        """
    )
    parser.add_argument("topic", help="Essay topic")
    parser.add_argument("--pages", type=int, default=10,
                        help="Target page count")
    parser.add_argument("--style", default="accessible",
                        choices=["academic", "accessible", "narrative", "popular"],
                        help="Writing style")
    parser.add_argument("--autonomy", default="full",
                        choices=["full", "supervised", "interactive"],
                        help="Autonomy level (legacy, use --delegate-drafting instead)")
    parser.add_argument("--evidence-first", action="store_true",
                        help="Require evidence before any claims")
    parser.add_argument("--strict", "--strict-library-only", action="store_true",
                        dest="strict",
                        help="Library sources only, no web")
    parser.add_argument("--delegate-drafting", action="store_true",
                        help="Have CLI generate draft using internal LLM (default: return brief for Claude Code)")
    parser.add_argument("--format", choices=["json", "text"], default="json",
                        help="Output format")

    args = parser.parse_args()

    try:
        result = run_essay_workflow(
            topic=args.topic,
            pages=args.pages,
            style=args.style,
            autonomy=args.autonomy,
            evidence_first=args.evidence_first,
            strict=args.strict,
            delegate_drafting=args.delegate_drafting
        )

        if result.get('mode') == 'delegated':
            # Delegated mode response
            if result.get('draft_result', {}).get('success'):
                response = success_response(
                    f"Essay drafted: {args.topic}",
                    data=result
                )
                response.data["next_suggested_commands"] = [
                    f"cat {result['artifacts']['draft']}",
                    f"rdf validate {result['artifacts']['draft']}",
                ]
            else:
                response = error_response(
                    ErrorCodes.UNKNOWN_ERROR,
                    result.get('draft_result', {}).get('error', 'Drafting failed'),
                    data=result
                )
        else:
            # Brief mode response (default)
            response = success_response(
                f"Essay brief generated for: {args.topic}",
                data=result
            )
            response.data["next_suggested_commands"] = [
                "# Claude Code should write the essay using the brief above",
                f"# Save draft to: {result['project_dir']}/essay.md",
            ]

        if args.format == "json":
            response.print_json()
        else:
            print(f"Essay ID: {result['essay_id']}")
            print(f"Project: {result['project_dir']}")
            print(f"Mode: {result['mode']}")
            if result['mode'] == 'brief':
                print(f"\nBrief saved to: {result['artifacts']['brief']}")
                print("\nClaude Code should write the essay using this brief.")
            else:
                if result.get('draft_result', {}).get('success'):
                    print(f"\nDraft: {result['artifacts']['draft']}")

        return 0

    except Exception as e:
        resp = error_response(
            ErrorCodes.UNKNOWN_ERROR,
            str(e)
        )
        resp.print_json()
        return 1


if __name__ == "__main__":
    sys.exit(main())
