#!/usr/bin/env python3
"""
Async Database Utilities for TUI and Interactive Applications.

This module provides async wrappers for synchronous database operations,
preventing UI freezing in Textual-based applications.

Dashboard Integration:
    The main dashboard.py uses Textual's run_worker() pattern which handles
    threading automatically. This module provides an alternative async API
    that can be used in pure async contexts or for future dashboard enhancements.

    Current dashboard pattern (uses workers):
        self.run_worker(self._search_worker(query))
        async def _search_worker(self, query: str):
            results = hybrid_search_with_rerank(query)  # sync call in worker

    Alternative using this module (pure async):
        async def perform_search(self, query: str):
            results = await async_hybrid_search(query)  # non-blocking

Usage:
    from db_utils_async import async_db_query, async_hybrid_search

    # In a Textual Screen:
    class SearchScreen(Screen):
        async def perform_search(self, query: str):
            self.query_one("#status").update("Searching...")
            results = await async_hybrid_search(query, limit=20)
            self.display_results(results)

Note:
    These wrappers use asyncio.to_thread() to run synchronous psycopg2
    calls in a thread pool. For truly async database operations, consider
    migrating to psycopg3 with AsyncConnection.
"""

import asyncio
from functools import partial
from typing import Dict, List, Any, Optional, Callable
import logging

logger = logging.getLogger(__name__)


# =============================================================================
# GENERIC ASYNC WRAPPER
# =============================================================================

async def async_db_query(
    query_func: Callable,
    *args,
    **kwargs
) -> Any:
    """
    Run a synchronous database function in a thread pool.

    This prevents blocking the asyncio event loop (and thus the TUI)
    during long-running database operations.

    Args:
        query_func: The synchronous function to call
        *args: Positional arguments for the function
        **kwargs: Keyword arguments for the function

    Returns:
        The result of the synchronous function

    Example:
        # Instead of:
        results = hybrid_search_with_rerank(query="test", limit=10)

        # Use:
        results = await async_db_query(
            hybrid_search_with_rerank,
            query="test",
            limit=10
        )
    """
    return await asyncio.to_thread(partial(query_func, *args, **kwargs))


# =============================================================================
# CONVENIENCE WRAPPERS
# =============================================================================

async def async_hybrid_search(
    query: str,
    limit: int = 10,
    use_rerank: bool = True,
    document_filter: List[str] = None,
    concept_filter: List[str] = None
) -> List[Dict[str, Any]]:
    """
    Async wrapper for hybrid_search_with_rerank.

    Args:
        query: Search query text
        limit: Maximum results to return
        use_rerank: Whether to use cross-encoder reranking
        document_filter: List of document_ids to filter results
        concept_filter: List of concept names to filter results

    Returns:
        List of search results with scores
    """
    from db_utils import hybrid_search_with_rerank

    return await async_db_query(
        hybrid_search_with_rerank,
        query_text=query,
        limit=limit,
        use_rerank=use_rerank,
        document_filter=document_filter,
        concept_filter=concept_filter
    )


async def async_graphrag_search(
    query: str,
    limit: int = 10,
    hop_depth: int = 2,
    include_direct_search: bool = True
) -> Dict[str, Any]:
    """
    Async wrapper for graphrag_search.

    Args:
        query: Search query text
        limit: Maximum results per search type
        hop_depth: Graph traversal depth
        include_direct_search: Whether to include standard vector search

    Returns:
        Dict with 'results', 'graph_context', and 'query_concepts'
    """
    from db_utils import graphrag_search

    return await async_db_query(
        graphrag_search,
        query=query,
        limit=limit,
        hop_depth=hop_depth,
        include_direct_search=include_direct_search
    )


async def async_execute_query(
    query: str,
    params: tuple = None,
    fetch: str = 'all'
) -> Any:
    """
    Async wrapper for execute_query.

    Args:
        query: SQL query string
        params: Query parameters
        fetch: 'all', 'one', or 'none'

    Returns:
        Query results or None
    """
    from db_utils import execute_query

    return await async_db_query(
        execute_query,
        query,
        params,
        fetch=fetch
    )


async def async_get_document(document_id: str) -> Optional[Dict[str, Any]]:
    """
    Async fetch of a single document by ID.

    Args:
        document_id: Document ID to fetch

    Returns:
        Document dict or None if not found
    """
    return await async_execute_query(
        """
        SELECT d.*, a.name as author_name
        FROM documents d
        LEFT JOIN authors a ON d.author_id = a.author_id
        WHERE d.document_id = %s
        """,
        (document_id,),
        fetch='one'
    )


async def async_get_documents(
    limit: int = 100,
    offset: int = 0,
    status: str = None,
    search: str = None
) -> List[Dict[str, Any]]:
    """
    Async fetch of documents with optional filtering.

    Args:
        limit: Maximum documents to return
        offset: Pagination offset
        status: Filter by processing_status
        search: Search in title/author

    Returns:
        List of document dicts
    """
    conditions = []
    params = []

    if status:
        conditions.append("d.processing_status = %s")
        params.append(status)

    if search:
        conditions.append("(d.title ILIKE %s OR a.name ILIKE %s)")
        params.extend([f"%{search}%", f"%{search}%"])

    where_clause = f"WHERE {' AND '.join(conditions)}" if conditions else ""

    params.extend([limit, offset])

    return await async_execute_query(
        f"""
        SELECT d.document_id, d.title, a.name as author,
               d.publication_year, d.processing_status,
               d.created_at, d.is_pinned
        FROM documents d
        LEFT JOIN authors a ON d.author_id = a.author_id
        {where_clause}
        ORDER BY d.created_at DESC
        LIMIT %s OFFSET %s
        """,
        tuple(params),
        fetch='all'
    )


async def async_get_concepts(
    limit: int = 100,
    offset: int = 0,
    category: str = None
) -> List[Dict[str, Any]]:
    """
    Async fetch of concepts with optional filtering.

    Args:
        limit: Maximum concepts to return
        offset: Pagination offset
        category: Filter by category

    Returns:
        List of concept dicts with document counts
    """
    if category:
        return await async_execute_query(
            """
            SELECT c.concept_id, c.name, c.category, c.aliases,
                   COUNT(dc.document_id) as document_count
            FROM concepts c
            LEFT JOIN document_concepts dc ON c.concept_id = dc.concept_id
            WHERE c.category = %s
            GROUP BY c.concept_id
            ORDER BY document_count DESC
            LIMIT %s OFFSET %s
            """,
            (category, limit, offset),
            fetch='all'
        )
    else:
        return await async_execute_query(
            """
            SELECT c.concept_id, c.name, c.category, c.aliases,
                   COUNT(dc.document_id) as document_count
            FROM concepts c
            LEFT JOIN document_concepts dc ON c.concept_id = dc.concept_id
            GROUP BY c.concept_id
            ORDER BY document_count DESC
            LIMIT %s OFFSET %s
            """,
            (limit, offset),
            fetch='all'
        )


async def async_get_statistics() -> Dict[str, Any]:
    """
    Async fetch of database statistics for dashboard.

    Returns:
        Dict with document counts, chunk counts, concept counts, etc.
    """
    # Run multiple queries concurrently
    results = await asyncio.gather(
        async_execute_query(
            "SELECT COUNT(*) as count FROM documents",
            fetch='one'
        ),
        async_execute_query(
            "SELECT COUNT(*) as count FROM chunks",
            fetch='one'
        ),
        async_execute_query(
            "SELECT COUNT(*) as count FROM concepts",
            fetch='one'
        ),
        async_execute_query(
            "SELECT COUNT(*) as count FROM chunk_embeddings",
            fetch='one'
        ),
        async_execute_query(
            """
            SELECT processing_status, COUNT(*) as count
            FROM documents
            GROUP BY processing_status
            """,
            fetch='all'
        )
    )

    doc_count = results[0]['count'] if results[0] else 0
    chunk_count = results[1]['count'] if results[1] else 0
    concept_count = results[2]['count'] if results[2] else 0
    embedding_count = results[3]['count'] if results[3] else 0
    status_counts = {r['processing_status']: r['count'] for r in (results[4] or [])}

    return {
        'documents': doc_count,
        'chunks': chunk_count,
        'concepts': concept_count,
        'embeddings': embedding_count,
        'status_breakdown': status_counts,
        'embedding_coverage': round(embedding_count / chunk_count * 100, 1) if chunk_count > 0 else 0
    }


async def async_log_search(
    query: str,
    result_count: int,
    retrieval_method: str = 'async'
) -> None:
    """
    Async logging of search queries.

    Args:
        query: Search query text
        result_count: Number of results returned
        retrieval_method: Search method used
    """
    from db_utils import log_search

    await async_db_query(
        log_search,
        query=query,
        result_count=result_count,
        retrieval_method=retrieval_method
    )


# =============================================================================
# RESEARCH AGENT ASYNC SUPPORT
# =============================================================================

async def async_research(
    question: str,
    max_iterations: int = 5,
    budget_limit_usd: float = 1.0,
    use_graphrag: bool = True
) -> Dict[str, Any]:
    """
    Async wrapper for ResearchAgent.research().

    Runs the entire research session in a thread pool to prevent
    blocking the TUI during long-running research operations.

    Args:
        question: Research question
        max_iterations: Maximum research iterations
        budget_limit_usd: API budget limit
        use_graphrag: Whether to use GraphRAG

    Returns:
        Dict with research session results
    """
    from research_agent import ResearchAgent, format_json_report
    import json

    def run_research():
        agent = ResearchAgent(
            max_iterations=max_iterations,
            budget_limit_usd=budget_limit_usd,
            use_graphrag=use_graphrag
        )
        session = agent.research(question)
        return json.loads(format_json_report(session))

    return await asyncio.to_thread(run_research)


# =============================================================================
# UTILITY FUNCTIONS
# =============================================================================

class AsyncDBContext:
    """
    Async context manager for batching multiple database operations.

    Usage:
        async with AsyncDBContext() as ctx:
            docs = await ctx.get_documents(limit=10)
            concepts = await ctx.get_concepts(limit=10)
            stats = await ctx.get_statistics()
    """

    async def __aenter__(self):
        return self

    async def __aexit__(self, exc_type, exc_val, exc_tb):
        pass

    async def get_documents(self, **kwargs):
        return await async_get_documents(**kwargs)

    async def get_concepts(self, **kwargs):
        return await async_get_concepts(**kwargs)

    async def get_statistics(self):
        return await async_get_statistics()

    async def hybrid_search(self, **kwargs):
        return await async_hybrid_search(**kwargs)

    async def graphrag_search(self, **kwargs):
        return await async_graphrag_search(**kwargs)


if __name__ == '__main__':
    # Test async functions
    async def test():
        print("Testing async database utilities...")

        print("\n1. Testing async_get_statistics...")
        stats = await async_get_statistics()
        print(f"   Documents: {stats['documents']}")
        print(f"   Chunks: {stats['chunks']}")
        print(f"   Concepts: {stats['concepts']}")

        print("\n2. Testing async_get_documents...")
        docs = await async_get_documents(limit=3)
        for doc in docs:
            print(f"   - {doc.get('title', 'Untitled')}")

        print("\n3. Testing async_hybrid_search...")
        results = await async_hybrid_search("test query", limit=3)
        print(f"   Found {len(results)} results")

        print("\n4. Testing concurrent queries...")
        import time
        start = time.time()
        results = await asyncio.gather(
            async_get_statistics(),
            async_get_documents(limit=5),
            async_get_concepts(limit=5)
        )
        elapsed = time.time() - start
        print(f"   3 concurrent queries completed in {elapsed:.2f}s")

        print("\nAll tests passed!")

    asyncio.run(test())
