#!/usr/bin/env python3
"""
CLI Utilities - Common utilities for all CLI tools

Provides:
- Structured JSON error responses
- Undo logging for reversible operations
- Summary mode for token-efficient output
- Error code definitions
"""

import json
import sys
import traceback
from pathlib import Path
from datetime import datetime
from typing import Dict, Any, Optional, List, Callable
from dataclasses import dataclass, asdict
from functools import wraps

# Error codes for structured responses
class ErrorCodes:
    """Standard error codes for CLI tools."""

    # General errors
    SUCCESS = "SUCCESS"
    UNKNOWN_ERROR = "UNKNOWN_ERROR"
    INVALID_ARGUMENT = "INVALID_ARGUMENT"

    # File errors
    FILE_NOT_FOUND = "FILE_NOT_FOUND"
    FILE_UNREADABLE = "FILE_UNREADABLE"
    FILE_CORRUPT = "FILE_CORRUPT"

    # Document errors
    DOCUMENT_NOT_FOUND = "DOCUMENT_NOT_FOUND"
    CORRUPT_PDF = "CORRUPT_PDF"
    OCR_FAILED = "OCR_FAILED"
    EMPTY_CONTENT = "EMPTY_CONTENT"

    # Database errors
    DATABASE_ERROR = "DATABASE_ERROR"
    CONNECTION_FAILED = "CONNECTION_FAILED"
    QUERY_FAILED = "QUERY_FAILED"

    # API errors
    API_KEY_MISSING = "API_KEY_MISSING"
    API_RATE_LIMITED = "API_RATE_LIMITED"
    API_ERROR = "API_ERROR"
    BUDGET_EXCEEDED = "BUDGET_EXCEEDED"

    # Workflow errors
    PHASE_NOT_COMPLETED = "PHASE_NOT_COMPLETED"
    INSUFFICIENT_RESEARCH = "INSUFFICIENT_RESEARCH"
    GAP_THRESHOLD_NOT_MET = "GAP_THRESHOLD_NOT_MET"
    PROJECT_NOT_FOUND = "PROJECT_NOT_FOUND"


# Actionable advice mapping
ERROR_ADVICE = {
    ErrorCodes.CORRUPT_PDF: "Run: python pipeline/reocr_document.py {document_id} --method easyocr",
    ErrorCodes.OCR_FAILED: "Run: python pipeline/reocr_document.py {document_id} --method tesseract",
    ErrorCodes.FILE_NOT_FOUND: "Check the file path exists and is accessible",
    ErrorCodes.DOCUMENT_NOT_FOUND: "Run: python pipeline/search_export.py '{query}' to find similar documents",
    ErrorCodes.DATABASE_ERROR: "Check database connection with: python pipeline/db_utils.py --test",
    ErrorCodes.API_KEY_MISSING: "Set environment variable: export OPENAI_API_KEY=your_key",
    ErrorCodes.API_RATE_LIMITED: "Wait 60 seconds and retry, or reduce --max-iterations",
    ErrorCodes.BUDGET_EXCEEDED: "Increase budget with --budget flag or reduce scope",
    ErrorCodes.INSUFFICIENT_RESEARCH: "Run: python pipeline/research_agent.py '{query}' to gather more sources",
    ErrorCodes.GAP_THRESHOLD_NOT_MET: "Review gaps.md and run Phase 3 with --auto-fill-gaps",
    ErrorCodes.PHASE_NOT_COMPLETED: "Run previous phases first or use --resume",
    ErrorCodes.PROJECT_NOT_FOUND: "Run: python pipeline/book.py --list-projects",
}


@dataclass
class StructuredResponse:
    """Structured response for CLI tools (success or error)."""
    status: str  # "success" or "error"
    code: str    # Error code
    message: str
    data: Optional[Dict[str, Any]] = None
    actionable_advice: Optional[str] = None
    details: Optional[Dict[str, Any]] = None

    def to_json(self, indent: int = 2) -> str:
        """Convert to JSON string."""
        output = {
            "status": self.status,
            "code": self.code,
            "message": self.message,
        }
        if self.data:
            output["data"] = self.data
        if self.actionable_advice:
            output["actionable_advice"] = self.actionable_advice
        if self.details:
            output["details"] = self.details
        return json.dumps(output, indent=indent, default=str)

    def print_json(self) -> None:
        """Print as JSON to stdout."""
        print(self.to_json())


def success_response(
    message: str,
    data: Optional[Dict[str, Any]] = None,
    code: str = ErrorCodes.SUCCESS
) -> StructuredResponse:
    """Create a success response."""
    return StructuredResponse(
        status="success",
        code=code,
        message=message,
        data=data
    )


def error_response(
    code: str,
    message: str,
    details: Optional[Dict[str, Any]] = None,
    context: Optional[Dict[str, str]] = None
) -> StructuredResponse:
    """Create an error response with actionable advice."""
    # Get advice and substitute context variables
    advice = ERROR_ADVICE.get(code, "")
    if advice and context:
        try:
            advice = advice.format(**context)
        except KeyError:
            pass  # Keep template if context missing

    return StructuredResponse(
        status="error",
        code=code,
        message=message,
        actionable_advice=advice or None,
        details=details
    )


def handle_cli_errors(json_output: bool = False):
    """
    Decorator for CLI main functions to catch and format errors.

    Usage:
        @handle_cli_errors(json_output=args.format == 'json')
        def main():
            ...
    """
    def decorator(func: Callable):
        @wraps(func)
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except FileNotFoundError as e:
                resp = error_response(
                    ErrorCodes.FILE_NOT_FOUND,
                    f"File not found: {e.filename or str(e)}",
                    details={"exception": str(e)}
                )
                if json_output:
                    resp.print_json()
                else:
                    print(f"Error: {resp.message}")
                    if resp.actionable_advice:
                        print(f"Suggestion: {resp.actionable_advice}")
                return 1
            except PermissionError as e:
                resp = error_response(
                    ErrorCodes.FILE_UNREADABLE,
                    f"Permission denied: {e.filename or str(e)}",
                    details={"exception": str(e)}
                )
                if json_output:
                    resp.print_json()
                else:
                    print(f"Error: {resp.message}")
                return 1
            except Exception as e:
                # Map known exceptions to error codes
                error_code = ErrorCodes.UNKNOWN_ERROR
                if "corrupt" in str(e).lower() or "pdf" in str(e).lower():
                    error_code = ErrorCodes.CORRUPT_PDF
                elif "database" in str(e).lower() or "connection" in str(e).lower():
                    error_code = ErrorCodes.DATABASE_ERROR
                elif "api" in str(e).lower() or "rate limit" in str(e).lower():
                    error_code = ErrorCodes.API_RATE_LIMITED
                elif "budget" in str(e).lower():
                    error_code = ErrorCodes.BUDGET_EXCEEDED

                resp = error_response(
                    error_code,
                    str(e),
                    details={
                        "exception_type": type(e).__name__,
                        "traceback": traceback.format_exc() if json_output else None
                    }
                )
                if json_output:
                    resp.print_json()
                else:
                    print(f"Error: {resp.message}")
                    if resp.actionable_advice:
                        print(f"Suggestion: {resp.actionable_advice}")
                return 1
        return wrapper
    return decorator


# =============================================================================
# UNDO LOGGING
# =============================================================================

@dataclass
class UndoLogEntry:
    """A single undo log entry."""
    timestamp: str
    operation: str  # "update_metadata", "merge_authors", "delete_document", etc.
    table: str
    affected_ids: List[str]
    before_state: Dict[str, Any]
    after_state: Dict[str, Any]
    user: str = "auto"
    notes: str = ""


class UndoLog:
    """
    Manages undo logging for reversible operations.

    Usage:
        undo_log = UndoLog(Path("./undo_log.json"))

        # Log a change
        undo_log.log_change(
            operation="update_metadata",
            table="documents",
            affected_ids=["DOC_001"],
            before_state={"author": "Old Author"},
            after_state={"author": "New Author"}
        )

        # Get undo command
        undo_log.get_last_entry()

        # Undo last change
        undo_log.undo_last()
    """

    def __init__(self, log_path: Path):
        self.log_path = log_path
        self.entries: List[UndoLogEntry] = []
        self._load()

    def _load(self) -> None:
        """Load existing log entries."""
        if self.log_path.exists():
            try:
                with open(self.log_path, 'r') as f:
                    data = json.load(f)
                    self.entries = [
                        UndoLogEntry(**entry) for entry in data.get('entries', [])
                    ]
            except (json.JSONDecodeError, KeyError):
                self.entries = []

    def _save(self) -> None:
        """Save log to file."""
        data = {
            'version': 1,
            'last_updated': datetime.now().isoformat(),
            'entries': [asdict(e) for e in self.entries[-100:]]  # Keep last 100
        }
        self.log_path.parent.mkdir(parents=True, exist_ok=True)
        with open(self.log_path, 'w') as f:
            json.dump(data, f, indent=2, default=str)

    def log_change(
        self,
        operation: str,
        table: str,
        affected_ids: List[str],
        before_state: Dict[str, Any],
        after_state: Dict[str, Any],
        notes: str = ""
    ) -> UndoLogEntry:
        """Log a reversible change."""
        entry = UndoLogEntry(
            timestamp=datetime.now().isoformat(),
            operation=operation,
            table=table,
            affected_ids=affected_ids,
            before_state=before_state,
            after_state=after_state,
            notes=notes
        )
        self.entries.append(entry)
        self._save()
        return entry

    def get_last_entry(self) -> Optional[UndoLogEntry]:
        """Get the most recent log entry."""
        return self.entries[-1] if self.entries else None

    def get_entries(self, limit: int = 10) -> List[UndoLogEntry]:
        """Get recent log entries."""
        return self.entries[-limit:]

    def get_undo_sql(self, entry: Optional[UndoLogEntry] = None) -> List[str]:
        """Generate SQL to undo a change."""
        entry = entry or self.get_last_entry()
        if not entry:
            return []

        sql_statements = []

        if entry.operation == "update_metadata":
            # Restore previous metadata values
            for doc_id in entry.affected_ids:
                for field, value in entry.before_state.items():
                    if value is None:
                        sql_statements.append(
                            f"UPDATE {entry.table} SET {field} = NULL WHERE document_id = '{doc_id}';"
                        )
                    else:
                        escaped_value = str(value).replace("'", "''")
                        sql_statements.append(
                            f"UPDATE {entry.table} SET {field} = '{escaped_value}' WHERE document_id = '{doc_id}';"
                        )

        elif entry.operation == "merge_authors":
            # Cannot easily undo author merges - need manual intervention
            sql_statements.append(
                f"-- Author merge cannot be auto-undone. Original authors: {entry.before_state}"
            )

        elif entry.operation == "quarantine_document":
            # Restore to active status
            for doc_id in entry.affected_ids:
                sql_statements.append(
                    f"UPDATE documents SET curation_status = 'active', quarantined_at = NULL WHERE document_id = '{doc_id}';"
                )

        return sql_statements

    def pop_last(self) -> Optional[UndoLogEntry]:
        """Remove and return the last entry."""
        if self.entries:
            entry = self.entries.pop()
            self._save()
            return entry
        return None


# =============================================================================
# SUMMARY MODE UTILITIES
# =============================================================================

def format_summary_results(
    results: List[Dict[str, Any]],
    fields: List[str] = None,
    max_results: int = 20
) -> List[Dict[str, Any]]:
    """
    Format results for summary mode (minimal fields for token efficiency).

    Args:
        results: Full result list
        fields: Fields to include (default: id, title, score)
        max_results: Maximum results to return

    Returns:
        Condensed result list
    """
    fields = fields or ['document_id', 'title', 'relevance_score']

    summary = []
    for i, r in enumerate(results[:max_results]):
        item = {}
        for field in fields:
            # Handle various field name conventions
            value = r.get(field) or r.get(field.replace('_', '')) or r.get(field.split('_')[0])
            if value is not None:
                # Truncate strings
                if isinstance(value, str) and len(value) > 80:
                    value = value[:77] + '...'
                item[field] = value
        if item:
            item['rank'] = i + 1
            summary.append(item)

    return summary


def format_summary_text(
    results: List[Dict[str, Any]],
    title_field: str = 'title',
    score_field: str = 'relevance_score'
) -> str:
    """
    Format results as compact text for summary mode.

    Returns one line per result: "1. Title (score)"
    """
    lines = []
    for i, r in enumerate(results[:20], 1):
        title = r.get(title_field, 'Untitled')[:60]
        score = r.get(score_field, 0)
        if isinstance(score, float):
            lines.append(f"{i}. {title} ({score:.2f})")
        else:
            lines.append(f"{i}. {title}")
    return '\n'.join(lines)


# =============================================================================
# EXPORTS
# =============================================================================

__all__ = [
    'ErrorCodes',
    'StructuredResponse',
    'success_response',
    'error_response',
    'handle_cli_errors',
    'UndoLog',
    'UndoLogEntry',
    'format_summary_results',
    'format_summary_text',
]
