#!/usr/bin/env python3
"""
RDF Capabilities - Agent Bootstrapping Meta-Tool

Returns a complete capability manifest optimized for Claude Code
to efficiently discover what RDF can do.

Usage:
    rdf capabilities --format json
"""

import argparse
import json
import sys
from pathlib import Path
from typing import Dict, Any, List

# Setup path for imports
sys.path.insert(0, str(Path(__file__).parent.parent))
sys.path.insert(0, str(Path(__file__).parent))

from pipeline.cli_utils import success_response, error_response, ErrorCodes


def get_version() -> str:
    """Get RDF version."""
    return "4.0.0"


def get_commands_manifest() -> Dict[str, Dict[str, Any]]:
    """Get comprehensive command manifest."""
    return {
        # Core Data Commands
        "ingest": {
            "description": "Add documents to library",
            "category": "data",
            "flags": ["--ocr-profile", "--force", "--dry-run", "--format"],
            "examples": [
                "rdf ingest NEW_DOCS/",
                "rdf ingest document.pdf --ocr-profile standard"
            ],
            "uses_llm": "embeddings_only",
            "output": "Indexed document with chunks and embeddings"
        },
        "search": {
            "description": "Semantic and keyword search",
            "category": "data",
            "flags": ["--limit", "--hybrid", "--keyword-only", "--semantic-only", "--format"],
            "examples": [
                "rdf search 'medieval guilds' --limit 20",
                "rdf search 'alchemy' --hybrid --format json"
            ],
            "uses_llm": "embeddings",
            "output": "Ranked search results with relevance scores"
        },
        "fetch": {
            "description": "Retrieve bounded chunk text by ID",
            "category": "data",
            "flags": ["--format"],
            "examples": [
                "rdf fetch CHUNK_001",
                "rdf fetch CHUNK_001 --format json"
            ],
            "uses_llm": False,
            "output": "Full chunk text with metadata"
        },
        "quotes": {
            "description": "Extract evidence passages for a topic",
            "category": "data",
            "flags": ["--limit", "--format"],
            "examples": [
                "rdf quotes 'Templar banking' --limit 10"
            ],
            "uses_llm": False,
            "output": "Relevant quotes with source citations"
        },

        # Action Commands
        "research": {
            "description": "Multi-iteration autonomous research",
            "category": "action",
            "flags": ["--iterations", "--strict-library-only", "--use-graphrag", "--format"],
            "examples": [
                "rdf research 'origins of Freemasonry'",
                "rdf research 'alchemical symbolism' --iterations 5"
            ],
            "uses_llm": "gpt-4o-mini (synthesis)",
            "output": "Research synthesis with sources and gaps"
        },
        "outline": {
            "description": "Generate book outline",
            "category": "action",
            "flags": ["--chapters", "--pages", "--style", "--format"],
            "examples": [
                "rdf outline 'History of Hermeticism' --chapters 5 --pages 100"
            ],
            "uses_llm": "gpt-4o",
            "output": "YAML outline with chapters and subjects"
        },
        "book": {
            "description": "Book compilation workflow (5 phases)",
            "category": "action",
            "flags": [
                "--phases", "--resume", "--delegate-drafting", "--evidence-first",
                "--writing-preset", "--polish-drafts", "--auto", "--format"
            ],
            "examples": [
                "rdf book outline.yaml --phases 1-4",
                "rdf book outline.yaml --phases 5 --delegate-drafting",
                "rdf book --resume BOOK_xxx"
            ],
            "uses_llm": "gpt-4o (with --delegate-drafting)",
            "output": "Chapter briefs (default) or drafted chapters (--delegate-drafting)",
            "opt_in_generation": {
                "flag": "--delegate-drafting",
                "default_behavior": "Returns chapter briefs for Claude Code to write",
                "delegated_behavior": "CLI generates drafts using internal LLM"
            }
        },
        "essay": {
            "description": "Essay generation",
            "category": "action",
            "flags": ["--pages", "--style", "--delegate-drafting", "--evidence-first", "--format"],
            "examples": [
                "rdf essay 'Topic' --pages 15",
                "rdf essay 'Topic' --pages 15 --delegate-drafting"
            ],
            "uses_llm": "gpt-4o (with --delegate-drafting)",
            "output": "Essay brief (default) or drafted essay (--delegate-drafting)",
            "opt_in_generation": {
                "flag": "--delegate-drafting",
                "default_behavior": "Returns essay brief for Claude Code to write",
                "delegated_behavior": "CLI generates essay using internal LLM"
            }
        },
        "write": {
            "description": "Universal entry point - infers essay vs book",
            "category": "action",
            "flags": ["--pages", "--chapters", "--type", "--style", "--format"],
            "examples": [
                "rdf write 'Topic' --pages 20",
                "rdf write 'Topic' --chapters 5"
            ],
            "uses_llm": "depends_on_target",
            "output": "Dispatches to essay or book command"
        },

        # Governance Commands
        "validate": {
            "description": "Verify claims against library sources",
            "category": "governance",
            "flags": ["--top-k", "--format"],
            "examples": [
                "rdf validate chapter_01.md",
                "rdf validate draft.md --format json"
            ],
            "uses_llm": "gpt-4o-mini",
            "output": "Validation report with verified/unverified claims"
        },
        "polish": {
            "description": "Style refinement for drafts",
            "category": "action",
            "flags": [
                "--preset", "--style", "--reference-style",
                "--use-internal-model", "--validate", "--format"
            ],
            "examples": [
                "rdf polish chapter_01.md --preset academic",
                "rdf polish chapter_01.md --preset accessible --use-internal-model"
            ],
            "uses_llm": "gpt-4o (with --use-internal-model)",
            "output": "Suggestions (default) or polished text (--use-internal-model)",
            "opt_in_generation": {
                "flag": "--use-internal-model",
                "default_behavior": "Returns editing suggestions for Claude Code to apply",
                "delegated_behavior": "CLI rewrites text using internal LLM"
            }
        },
        "queue": {
            "description": "Review queue management",
            "category": "governance",
            "flags": ["--type", "--format"],
            "subcommands": ["list", "approve", "reject", "clear"],
            "examples": [
                "rdf queue list",
                "rdf queue list --type gap",
                "rdf queue approve GAP_001",
                "rdf queue reject GAP_002 --reason 'out_of_scope'"
            ],
            "uses_llm": False,
            "output": "Queue items with approval status"
        },

        # Utility Commands
        "status": {
            "description": "Show workflow state",
            "category": "utility",
            "flags": ["--last", "--format"],
            "examples": [
                "rdf status",
                "rdf status --format json"
            ],
            "uses_llm": False,
            "output": "Active workflows with phase and progress"
        },
        "context": {
            "description": "Agent session warm-start",
            "category": "utility",
            "flags": ["--list", "--format"],
            "examples": [
                "rdf context BOOK_xxx",
                "rdf context --list",
                "rdf context"
            ],
            "uses_llm": False,
            "output": "Context packet for session resumption",
            "purpose": "Single call replaces 5+ file reads for context reconstruction"
        },
        "health": {
            "description": "Library health scan",
            "category": "utility",
            "flags": ["--format"],
            "examples": [
                "rdf health",
                "rdf health --format json"
            ],
            "uses_llm": False,
            "output": "Library health metrics and issues"
        },
        "edit-meta": {
            "description": "Edit document metadata",
            "category": "utility",
            "flags": ["--title", "--author", "--year", "--format"],
            "examples": [
                "rdf edit-meta DOC_023 --title 'Correct Title'"
            ],
            "uses_llm": False,
            "output": "Updated metadata confirmation"
        },
        "assess": {
            "description": "Mechanical document assessment",
            "category": "utility",
            "flags": ["--format"],
            "examples": [
                "rdf assess DOC_023"
            ],
            "uses_llm": False,
            "output": "Quality metrics for document"
        },
        "graph": {
            "description": "Knowledge graph queries",
            "category": "utility",
            "flags": ["--format"],
            "subcommands": ["query", "entities", "relationships"],
            "examples": [
                "rdf graph query 'alchemy'",
                "rdf graph entities --type Person"
            ],
            "uses_llm": False,
            "output": "Graph data in JSON format"
        },
        "diff": {
            "description": "File comparison",
            "category": "utility",
            "flags": ["--format"],
            "examples": [
                "rdf diff file_a.md file_b.md"
            ],
            "uses_llm": False,
            "output": "Diff output"
        },
        "config": {
            "description": "Configuration management",
            "category": "utility",
            "subcommands": ["entity-extraction", "show"],
            "examples": [
                "rdf config entity-extraction",
                "rdf config show"
            ],
            "uses_llm": False,
            "output": "Configuration status"
        },
    }


def get_workflows() -> Dict[str, List[str]]:
    """Get standard workflow sequences."""
    return {
        "book": [
            "outline → generates outline.yaml",
            "book --phases 1-4 → research, gaps, synthesis",
            "book --phases 5 → chapter briefs (or drafts with --delegate-drafting)",
            "validate → verify claims",
            "polish → refine style",
            "compile → final manuscript"
        ],
        "essay": [
            "essay → essay brief (or draft with --delegate-drafting)",
            "validate → verify claims",
            "polish → refine style"
        ],
        "research": [
            "research → synthesis with sources and gaps",
            "queue list --type gap → review gaps",
            "queue approve/reject → manage gaps"
        ]
    }


def get_configuration_status() -> Dict[str, Any]:
    """Get current configuration status."""
    status = {}

    try:
        from config import (
            EMBEDDINGS_CONFIG, GENERATION_CONFIG,
            ENTITY_EXTRACTION_CONFIG, OPENAI_ENABLED
        )

        status['embeddings'] = {
            'configured': True,
            'provider': EMBEDDINGS_CONFIG.get('provider', 'unknown'),
            'model': EMBEDDINGS_CONFIG.get('model', 'unknown'),
            'status': 'ready' if OPENAI_ENABLED else 'needs_api_key'
        }

        status['generation'] = {
            'configured': True,
            'default_author': GENERATION_CONFIG.get('default_author', 'agent'),
            'primary_model': GENERATION_CONFIG.get('primary_model', 'gpt-4o'),
            'utility_model': GENERATION_CONFIG.get('utility_model', 'gpt-4o-mini'),
            'status': 'ready' if OPENAI_ENABLED else 'needs_api_key'
        }

        entity_config = ENTITY_EXTRACTION_CONFIG if ENTITY_EXTRACTION_CONFIG else {}
        status['entity_extraction'] = {
            'configured': entity_config.get('configured', False),
            'provider': entity_config.get('provider', 'not_configured'),
            'status': 'ready' if entity_config.get('configured') else 'run_rdf_config'
        }

    except ImportError as e:
        status['error'] = f"Could not load config: {e}"

    return status


def get_error_codes() -> Dict[str, str]:
    """Get common error codes and their meanings."""
    return {
        "SUCCESS": "Operation completed successfully",
        "NOT_FOUND": "Resource not found",
        "INVALID_INPUT": "Invalid input parameters",
        "LIBRARY_EMPTY": "Library has no documents - run rdf ingest first",
        "CONFIGURATION_REQUIRED": "Run rdf config to configure",
        "BUDGET_EXCEEDED": "API budget exhausted",
        "QUEUE_GATE_BLOCKED": "Workflow constraint - run required command first",
        "PAUSED_FOR_REVIEW": "Workflow paused awaiting human review",
        "LLM_UNAVAILABLE": "LLM not available - check API key",
        "UNKNOWN_ERROR": "Unexpected error occurred"
    }


def get_library_status() -> Dict[str, Any]:
    """Get basic library status."""
    status = {
        'documents': 0,
        'chunks': 0,
        'status': 'unknown'
    }

    try:
        from db_utils import get_db_connection
        with get_db_connection() as conn:
            with conn.cursor() as cur:
                cur.execute("SELECT COUNT(*) FROM documents")
                status['documents'] = cur.fetchone()[0]

                cur.execute("SELECT COUNT(*) FROM chunks")
                status['chunks'] = cur.fetchone()[0]

                status['status'] = 'healthy' if status['documents'] > 0 else 'empty'
    except Exception as e:
        status['status'] = 'error'
        status['error'] = str(e)

    return status


def generate_capabilities_manifest() -> Dict[str, Any]:
    """Generate complete capabilities manifest."""
    return {
        'version': get_version(),
        'api_version': '2025-01',

        'commands': get_commands_manifest(),

        'workflows': get_workflows(),

        'configuration': get_configuration_status(),

        'error_codes': get_error_codes(),

        'library_status': get_library_status(),

        'opt_in_generation': {
            'description': 'Commands default to returning data for Claude Code to process',
            'flags': {
                '--delegate-drafting': 'For book/essay - CLI generates drafts',
                '--use-internal-model': 'For polish - CLI rewrites text'
            },
            'default_behavior': 'Claude Code writes prose, CLI provides data'
        },

        'session_management': {
            'warm_start': 'rdf context [PROJECT_ID]',
            'status_check': 'rdf status',
            'project_list': 'rdf context --list'
        }
    }


def main():
    parser = argparse.ArgumentParser(
        description="Return capability manifest for agent bootstrapping",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Full capabilities manifest (JSON)
  rdf capabilities --format json

  # Text summary
  rdf capabilities --format text
        """
    )

    parser.add_argument(
        "--format", "-f",
        choices=["json", "text"],
        default="json",
        help="Output format"
    )

    args = parser.parse_args()

    try:
        manifest = generate_capabilities_manifest()

        response = success_response(
            "RDF Capabilities Manifest",
            data=manifest
        )

        if args.format == "json":
            response.print_json()
        else:
            print(f"\n{'='*60}")
            print(f"RDF CAPABILITIES v{manifest['version']}")
            print(f"{'='*60}")

            # Commands summary
            print("\nCOMMANDS:")
            commands = manifest['commands']
            categories = {}
            for name, info in commands.items():
                cat = info.get('category', 'other')
                if cat not in categories:
                    categories[cat] = []
                categories[cat].append((name, info.get('description', '')))

            for category, cmds in sorted(categories.items()):
                print(f"\n  {category.upper()}:")
                for name, desc in sorted(cmds):
                    print(f"    {name:15} {desc}")

            # Opt-in generation
            print("\nOPT-IN GENERATION:")
            print("  Default: Claude Code writes prose, CLI provides data")
            print("  --delegate-drafting: CLI generates drafts (book/essay)")
            print("  --use-internal-model: CLI rewrites text (polish)")

            # Configuration
            print("\nCONFIGURATION STATUS:")
            config = manifest['configuration']
            for key, info in config.items():
                if isinstance(info, dict):
                    status = info.get('status', 'unknown')
                    print(f"  {key}: {status}")

            # Library
            lib = manifest['library_status']
            print(f"\nLIBRARY: {lib.get('documents', 0)} documents, {lib.get('chunks', 0)} chunks ({lib.get('status', 'unknown')})")

        return 0

    except Exception as e:
        resp = error_response(
            ErrorCodes.UNKNOWN_ERROR,
            str(e)
        )
        if args.format == "json":
            resp.print_json()
        else:
            print(f"Error: {e}")
        return 1


if __name__ == "__main__":
    sys.exit(main())
