#!/usr/bin/env python3
"""
Unified Book Generation Tool

Single-command book generation from outline to compiled manuscript.
Designed for Claude Code to orchestrate complete book research workflows.

Usage:
    # Full automated book generation
    python book.py outline.yaml --auto

    # Step-by-step with control
    python book.py outline.yaml --phase 1-4
    python book.py --resume BOOK_xxx --phase 5 --compile

    # Dry run to see what would happen
    python book.py outline.yaml --dry-run

    # Status check (JSON for Claude Code)
    python book.py --status BOOK_xxx --format json

Output:
    Complete manuscript in MD, DOCX, and EPUB formats.
"""

import argparse
import json
import logging
import subprocess
import sys
from pathlib import Path
from datetime import datetime
from typing import Optional, Dict, Any, List

# Setup path for imports
sys.path.insert(0, str(Path(__file__).parent))

from config import LOGGING_CONFIG, BASE_DIR, BOOK_WORKFLOW_CONFIG

# Book workflow imports
from book_workflow_models import BookProject, list_book_projects
from book_workflow_input import parse_outline_file, validate_project
from book_workflow_output import generate_status_json, write_all_outputs
from book_workflow_phases import run_workflow

# Setup logging
LOG_LEVEL = LOGGING_CONFIG.get('level', 'INFO')
logging.basicConfig(
    level=getattr(logging, LOG_LEVEL, logging.INFO),
    format='%(asctime)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# Default projects directory
PROJECTS_DIR = BOOK_WORKFLOW_CONFIG.get('projects_dir', BASE_DIR / 'book_projects')


# =============================================================================
# COMPILATION
# =============================================================================

def compile_book(
    project: BookProject,
    formats: List[str] = None,
    include_toc: bool = True,
    include_bibliography: bool = True,
    output_dir: Optional[Path] = None
) -> Dict[str, Any]:
    """
    Compile chapter drafts into final manuscript formats.

    Args:
        project: BookProject with completed drafts
        formats: Output formats (md, docx, epub, pdf)
        include_toc: Whether to include table of contents
        include_bibliography: Whether to include bibliography
        output_dir: Output directory (defaults to project/output)

    Returns:
        Dictionary with compilation results
    """
    formats = formats or ['md', 'docx']

    project_dir = Path(project.project_dir)
    drafts_dir = project_dir / 'drafts'
    output_dir = output_dir or project_dir / 'output'
    output_dir.mkdir(parents=True, exist_ok=True)

    # Collect draft files
    draft_files = sorted(drafts_dir.glob('*.md')) if drafts_dir.exists() else []

    if not draft_files:
        return {
            'success': False,
            'error': 'No draft files found',
            'drafts_dir': str(drafts_dir)
        }

    logger.info(f"Found {len(draft_files)} chapter drafts")

    results = {
        'success': True,
        'chapters': len(draft_files),
        'formats': {},
        'output_dir': str(output_dir)
    }

    # Generate safe filename from title
    safe_title = ''.join(c for c in project.title if c.isalnum() or c in ' -_')[:50]
    safe_title = safe_title.replace(' ', '_').lower()
    base_filename = f"{safe_title}_{project.project_id[-6:]}"

    # Assemble markdown manuscript
    md_output = output_dir / f"{base_filename}.md"
    _assemble_markdown(project, draft_files, md_output, include_toc)
    results['formats']['md'] = str(md_output)

    # Count words
    with open(md_output, 'r') as f:
        word_count = len(f.read().split())
    results['word_count'] = word_count

    # Generate other formats using pandoc if available
    if _pandoc_available():
        if 'docx' in formats:
            docx_output = output_dir / f"{base_filename}.docx"
            success = _generate_docx(md_output, docx_output, include_toc)
            if success:
                results['formats']['docx'] = str(docx_output)

        if 'epub' in formats:
            epub_output = output_dir / f"{base_filename}.epub"
            success = _generate_epub(md_output, epub_output, project.title, include_toc)
            if success:
                results['formats']['epub'] = str(epub_output)

        if 'pdf' in formats and _pdflatex_available():
            pdf_output = output_dir / f"{base_filename}.pdf"
            success = _generate_pdf(md_output, pdf_output, include_toc)
            if success:
                results['formats']['pdf'] = str(pdf_output)
    else:
        logger.warning("Pandoc not available - only markdown output generated")

    # Generate bibliography if requested
    if include_bibliography:
        bib_output = output_dir / f"{base_filename}_bibliography.md"
        _generate_bibliography(project, bib_output)
        results['bibliography'] = str(bib_output)

    return results


def _assemble_markdown(
    project: BookProject,
    draft_files: List[Path],
    output_path: Path,
    include_toc: bool
) -> None:
    """Assemble markdown from chapter drafts."""
    with open(output_path, 'w') as f:
        # Title page
        f.write(f"---\n")
        f.write(f"title: \"{project.title}\"\n")
        if project.author:
            f.write(f"author: \"{project.author}\"\n")
        f.write(f"date: \"{datetime.now().strftime('%Y-%m-%d')}\"\n")
        f.write(f"---\n\n")

        f.write(f"# {project.title}\n\n")
        if project.author:
            f.write(f"*by {project.author}*\n\n")

        f.write("---\n\n")

        # Table of contents
        if include_toc:
            f.write("## Table of Contents\n\n")
            for i, chapter in enumerate(project.chapters, 1):
                f.write(f"{i}. [{chapter.title}](#chapter-{i})\n")
            f.write("\n---\n\n")

        # Chapter content
        for i, draft_file in enumerate(draft_files, 1):
            with open(draft_file, 'r') as df:
                content = df.read()

            # Add chapter anchor
            f.write(f"<a name=\"chapter-{i}\"></a>\n\n")

            # Strip YAML front matter if present
            if content.startswith('---'):
                parts = content.split('---', 2)
                if len(parts) >= 3:
                    content = parts[2].strip()

            f.write(content)
            f.write("\n\n---\n\n")

    logger.info(f"Assembled markdown: {output_path}")


def _pandoc_available() -> bool:
    """Check if pandoc is installed."""
    try:
        subprocess.run(['pandoc', '--version'], capture_output=True, check=True)
        return True
    except (subprocess.CalledProcessError, FileNotFoundError):
        return False


def _pdflatex_available() -> bool:
    """Check if pdflatex is installed."""
    try:
        subprocess.run(['pdflatex', '--version'], capture_output=True, check=True)
        return True
    except (subprocess.CalledProcessError, FileNotFoundError):
        return False


def _generate_docx(input_path: Path, output_path: Path, include_toc: bool) -> bool:
    """Generate DOCX using pandoc."""
    cmd = [
        'pandoc', str(input_path),
        '-f', 'markdown+smart',
        '-t', 'docx',
        '-o', str(output_path),
        '--standalone',
        '--highlight-style=tango',
    ]
    if include_toc:
        cmd.extend(['--toc', '--toc-depth=2'])

    try:
        subprocess.run(cmd, capture_output=True, check=True)
        logger.info(f"Generated DOCX: {output_path}")
        return True
    except subprocess.CalledProcessError as e:
        logger.error(f"DOCX generation failed: {e.stderr.decode()}")
        return False


def _generate_epub(input_path: Path, output_path: Path, title: str, include_toc: bool) -> bool:
    """Generate EPUB using pandoc."""
    cmd = [
        'pandoc', str(input_path),
        '-f', 'markdown+smart',
        '-t', 'epub3',
        '-o', str(output_path),
        '--standalone',
        f'--metadata=title:{title}',
    ]
    if include_toc:
        cmd.extend(['--toc', '--toc-depth=2'])

    try:
        subprocess.run(cmd, capture_output=True, check=True)
        logger.info(f"Generated EPUB: {output_path}")
        return True
    except subprocess.CalledProcessError as e:
        logger.error(f"EPUB generation failed: {e.stderr.decode()}")
        return False


def _generate_pdf(input_path: Path, output_path: Path, include_toc: bool) -> bool:
    """Generate PDF using pandoc + pdflatex."""
    cmd = [
        'pandoc', str(input_path),
        '-f', 'markdown+smart',
        '-o', str(output_path),
        '--standalone',
        '--pdf-engine=pdflatex',
    ]
    if include_toc:
        cmd.extend(['--toc', '--toc-depth=2'])

    try:
        subprocess.run(cmd, capture_output=True, check=True)
        logger.info(f"Generated PDF: {output_path}")
        return True
    except subprocess.CalledProcessError as e:
        logger.warning(f"PDF generation failed: {e.stderr.decode()[:200]}")
        return False


def _generate_bibliography(project: BookProject, output_path: Path) -> None:
    """Generate bibliography from sources used in research."""
    # Collect unique sources
    sources = {}
    for chapter in project.chapters:
        for source in chapter.sources_used:
            doc_id = source.get('document_id')
            if doc_id and doc_id not in sources:
                sources[doc_id] = source

    if not sources:
        return

    with open(output_path, 'w') as f:
        f.write("# Bibliography\n\n")
        f.write(f"*{project.title}*\n\n")
        f.write("---\n\n")

        # Sort by author/title
        sorted_sources = sorted(
            sources.values(),
            key=lambda x: (x.get('author', 'Unknown'), x.get('title', ''))
        )

        for source in sorted_sources:
            author = source.get('author', 'Unknown')
            title = source.get('title', 'Untitled')
            year = source.get('publication_year', '')

            # Format: Author. *Title*. Year.
            f.write(f"- {author}. *{title}*.")
            if year:
                f.write(f" {year}.")
            f.write("\n")

    logger.info(f"Generated bibliography: {output_path}")


# =============================================================================
# UNIFIED WORKFLOW
# =============================================================================

def run_full_book_workflow(
    outline_path: Optional[Path] = None,
    project_id: Optional[str] = None,
    auto: bool = False,
    phases: str = '1-5',
    compile_output: bool = True,
    output_formats: List[str] = None,
    tavily_budget: int = 50,
    approve_all_gaps: bool = False,
    approve_priority: Optional[str] = None,
    dry_run: bool = False,
    verbose: bool = False,
    evidence_first: bool = False,
    gap_threshold: float = 0.0,
    writing_style: Optional[str] = None,
    writing_preset: Optional[str] = None,
    writing_prompt_file: Optional[str] = None,
    reference_voice: Optional[str] = None,
    polish_drafts: bool = False,
    delegate_drafting: bool = False
) -> Dict[str, Any]:
    """
    Run the complete book generation workflow.

    Args:
        outline_path: Path to YAML/JSON outline (for new projects)
        project_id: Project ID (for resuming existing)
        auto: Enable full automation (approve all gaps, compile)
        phases: Phase range to run (e.g., "1-5", "3", "4-5")
        compile_output: Whether to compile final output
        output_formats: Output formats for compilation
        tavily_budget: Credit budget for Tavily searches
        approve_all_gaps: Auto-approve all gaps
        approve_priority: Auto-approve gaps by priority
        dry_run: Preview without executing
        verbose: Show detailed progress
        evidence_first: Use evidence-first generation (quote-binning, no hallucination)
        gap_threshold: Minimum research coverage % before drafting (0-100, 0=disabled)
        writing_style: Custom writing style description (free text)
        writing_preset: Predefined writing style preset name
        writing_prompt_file: Path to file with custom style prompt
        reference_voice: Path to text file for voice cloning
        polish_drafts: Apply polishing pass after draft generation
        delegate_drafting: If True, CLI generates drafts using internal LLM.
                          If False (default), returns chapter briefs for Claude Code.

    Returns:
        Workflow results dictionary
    """
    output_formats = output_formats or ['md', 'docx', 'epub']

    # Parse phase range
    if '-' in phases:
        start, end = phases.split('-')
        start_phase, end_phase = int(start), int(end)
    else:
        start_phase = end_phase = int(phases)

    # Auto mode settings
    if auto:
        approve_all_gaps = True
        compile_output = True

    # Load or create project
    if project_id:
        project = _load_project(project_id)
        if not project:
            return {'success': False, 'error': f'Project not found: {project_id}'}
    elif outline_path:
        project = _create_project_from_outline(outline_path)
        if not project:
            return {'success': False, 'error': 'Failed to parse outline'}
    else:
        return {'success': False, 'error': 'Must provide outline_path or project_id'}

    logger.info(f"Book: {project.title}")
    logger.info(f"Chapters: {len(project.chapters)}")
    logger.info(f"Phases: {start_phase} to {end_phase}")

    if dry_run:
        return _dry_run_preview(project, start_phase, end_phase, compile_output, output_formats)

    # Handle gap approval
    if approve_all_gaps:
        for gap in project.all_gaps:
            gap['approved'] = True
        logger.info(f"Auto-approved all {len(project.all_gaps)} gaps")
    elif approve_priority:
        priority_levels = {
            'high': ['high'],
            'medium': ['high', 'medium'],
            'low': ['high', 'medium', 'low'],
            'all': ['high', 'medium', 'low']
        }
        approved_priorities = priority_levels.get(approve_priority, [])
        count = 0
        for gap in project.all_gaps:
            if gap.get('priority', 'medium') in approved_priorities:
                gap['approved'] = True
                count += 1
        logger.info(f"Auto-approved {count} gaps (priority: {approve_priority})")

    # Save project
    project_dir = PROJECTS_DIR / project.project_id
    project_dir.mkdir(parents=True, exist_ok=True)
    project.project_dir = str(project_dir)
    project.save(project_dir)

    results = {
        'project_id': project.project_id,
        'title': project.title,
        'phases_run': [],
        'success': True
    }

    # Run workflow phases
    def progress_callback(message: str, progress: float):
        if verbose:
            print(f"  {message} ({progress:.0%})")

    workflow_result = run_workflow(
        project,
        start_phase=start_phase,
        end_phase=end_phase,
        progress_callback=progress_callback,
        tavily_budget=tavily_budget,
        auto_fill_gaps=approve_all_gaps,
        generate_drafts=(end_phase >= 5),
        evidence_first=evidence_first,
        gap_threshold=gap_threshold,
        writing_style=writing_style,
        writing_preset=writing_preset,
        writing_prompt_file=writing_prompt_file,
        reference_voice=reference_voice,
        polish_drafts=polish_drafts,
        delegate_drafting=delegate_drafting,
    )

    results['phases_run'] = list(workflow_result.get('phases', {}).keys())
    results['workflow'] = workflow_result

    # Check for gap threshold halt
    if workflow_result.get('halted_for_gaps'):
        results['success'] = False
        results['halted_for_gaps'] = True
        results['gap_coverage'] = workflow_result.get('gap_coverage', 0)
        results['required_coverage'] = gap_threshold
        results['pending_research_queries'] = workflow_result.get('pending_research_queries', [])
        results['next_action'] = f"Research coverage {results['gap_coverage']:.0f}% is below threshold {gap_threshold:.0f}%. Fill gaps before drafting."
        return results

    if not workflow_result.get('success', False):
        results['success'] = False
        return results

    # Write outputs after workflow
    write_all_outputs(project, project_dir)

    # Compile if requested and drafts exist
    if compile_output and end_phase >= 5:
        logger.info("Compiling final manuscript...")
        compile_result = compile_book(
            project,
            formats=output_formats,
            include_toc=True,
            include_bibliography=True
        )
        results['compilation'] = compile_result

        if compile_result.get('success'):
            results['output_files'] = compile_result.get('formats', {})
            results['word_count'] = compile_result.get('word_count')
        else:
            logger.warning(f"Compilation had issues: {compile_result.get('error')}")

    # Final save
    project.save(project_dir)

    results['project_dir'] = str(project_dir)
    return results


def _load_project(project_id: str) -> Optional[BookProject]:
    """Load an existing project by ID."""
    project_dir = PROJECTS_DIR / project_id
    if not project_dir.exists():
        return None

    return BookProject.load(project_dir)


def _create_project_from_outline(outline_path: Path) -> Optional[BookProject]:
    """Create a new project from outline file."""
    try:
        project = parse_outline_file(outline_path)
        errors = validate_project(project)
        if errors:
            for err in errors:
                logger.error(err)
            return None
        return project
    except Exception as e:
        logger.error(f"Failed to parse outline: {e}")
        return None


def _dry_run_preview(
    project: BookProject,
    start_phase: int,
    end_phase: int,
    compile_output: bool,
    output_formats: List[str]
) -> Dict[str, Any]:
    """Generate dry-run preview without executing."""
    subjects = []
    for chapter in project.chapters:
        for subject in chapter.subjects:
            subjects.append(subject.topic)

    return {
        'dry_run': True,
        'project_id': project.project_id,
        'title': project.title,
        'author': project.author,
        'chapters': len(project.chapters),
        'subjects': subjects,
        'phases': f"{start_phase}-{end_phase}",
        'will_compile': compile_output,
        'output_formats': output_formats,
        'estimated_operations': {
            'library_searches': len(subjects) * 5,  # ~5 iterations per subject
            'gap_analysis': len(subjects),
            'synthesis': len(project.chapters),
            'draft_generation': len(project.chapters) if end_phase >= 5 else 0,
        }
    }


# =============================================================================
# CLI
# =============================================================================

def create_parser() -> argparse.ArgumentParser:
    """Create CLI argument parser."""
    parser = argparse.ArgumentParser(
        prog='book',
        description='Unified book generation from outline to manuscript',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Full automated book generation
  python book.py outline.yaml --auto

  # Research only (phases 1-4), compile later
  python book.py outline.yaml --phases 1-4
  python book.py --resume BOOK_xxx --phases 5 --compile

  # Dry run to see plan
  python book.py outline.yaml --dry-run

  # Check status (for Claude Code)
  python book.py --status BOOK_xxx --format json

  # Resume with gap approval
  python book.py --resume BOOK_xxx --phases 3-5 --approve-all-gaps
        """
    )

    # Input
    parser.add_argument(
        'outline',
        nargs='?',
        type=str,
        help='Path to YAML/JSON book outline'
    )

    # Project control
    control = parser.add_argument_group('Project Control')
    control.add_argument(
        '--resume', '-r',
        type=str,
        metavar='PROJECT_ID',
        help='Resume existing project'
    )
    control.add_argument(
        '--status', '-s',
        type=str,
        metavar='PROJECT_ID',
        help='Show project status'
    )
    control.add_argument(
        '--list-projects',
        action='store_true',
        help='List all book projects'
    )

    # Workflow options
    workflow = parser.add_argument_group('Workflow Options')
    workflow.add_argument(
        '--auto',
        action='store_true',
        help='Full automation (approve gaps, compile)'
    )
    workflow.add_argument(
        '--phases',
        type=str,
        default='1-5',
        help='Phases to run (e.g., "1-5", "3", "4-5")'
    )
    workflow.add_argument(
        '--dry-run',
        action='store_true',
        help='Preview without executing'
    )
    workflow.add_argument(
        '--evidence-first',
        action='store_true',
        help='Use evidence-first generation: quote-binning workflow that prevents hallucination'
    )
    workflow.add_argument(
        '--gap-threshold',
        type=float,
        default=0.0,
        metavar='PERCENT',
        help='Minimum research coverage %% before drafting (0-100). If below threshold, halts and requests more research.'
    )

    # Gap approval
    gaps = parser.add_argument_group('Gap Approval')
    gaps.add_argument(
        '--approve-all-gaps',
        action='store_true',
        help='Auto-approve all research gaps'
    )
    gaps.add_argument(
        '--approve-priority',
        type=str,
        choices=['high', 'medium', 'low', 'all'],
        help='Auto-approve gaps by priority level'
    )
    gaps.add_argument(
        '--tavily-budget',
        type=int,
        default=50,
        help='Tavily credit budget for gap filling'
    )

    # Compilation
    compile_grp = parser.add_argument_group('Compilation')
    compile_grp.add_argument(
        '--compile',
        action='store_true',
        help='Compile final manuscript'
    )
    compile_grp.add_argument(
        '--formats',
        type=str,
        default='md,docx,epub',
        help='Output formats (comma-separated)'
    )
    compile_grp.add_argument(
        '--no-toc',
        action='store_true',
        help='Exclude table of contents'
    )
    compile_grp.add_argument(
        '--no-bibliography',
        action='store_true',
        help='Exclude bibliography'
    )

    # Writing Style
    style_grp = parser.add_argument_group('Writing Style (Phase 5 Draft Generation)')
    style_exclusive = style_grp.add_mutually_exclusive_group()
    style_exclusive.add_argument(
        '--writing-style',
        type=str,
        metavar='DESCRIPTION',
        help='Custom writing style description (free text prompt)'
    )
    style_exclusive.add_argument(
        '--writing-preset',
        type=str,
        choices=[
            'academic', 'accessible', 'narrative', 'concise', 'popular',
            'technical', 'journalistic', 'literary', 'textbook', 'conversational'
        ],
        help='Use a predefined writing style preset'
    )
    style_exclusive.add_argument(
        '--writing-prompt-file',
        type=str,
        metavar='FILE',
        help='Path to file containing custom writing style prompt'
    )
    style_exclusive.add_argument(
        '--reference-voice',
        type=str,
        metavar='FILE',
        help='Path to text file whose writing style should be mimicked (voice cloning)'
    )
    style_grp.add_argument(
        '--polish-drafts',
        action='store_true',
        help='Apply polishing pass after draft generation'
    )
    style_grp.add_argument(
        '--delegate-drafting',
        action='store_true',
        help='Have CLI generate drafts using internal LLM (default: return chapter briefs for Claude Code)'
    )

    # Output
    output = parser.add_argument_group('Output')
    output.add_argument(
        '--format', '-f',
        type=str,
        choices=['text', 'json'],
        default='text',
        help='Output format for status/results'
    )
    output.add_argument(
        '--verbose', '-v',
        action='store_true',
        help='Show detailed progress'
    )
    output.add_argument(
        '--quiet', '-q',
        action='store_true',
        help='Minimal output'
    )

    return parser


def main():
    """Main entry point."""
    parser = create_parser()
    args = parser.parse_args()

    # List projects
    if args.list_projects:
        projects = list_book_projects(PROJECTS_DIR)
        if args.format == 'json':
            print(json.dumps(projects, indent=2, default=str))
        else:
            if not projects:
                print("No book projects found.")
            else:
                print(f"\n{'='*60}")
                print("BOOK PROJECTS")
                print(f"{'='*60}")
                for p in projects:
                    print(f"\n{p['project_id']}")
                    print(f"  Title: {p['title']}")
                    print(f"  Status: {p['status']}")
                    print(f"  Phase: {p['current_phase']}")
                    print(f"  Chapters: {p['chapter_count']}")
        return 0

    # Status check
    if args.status:
        project = _load_project(args.status)
        if not project:
            print(f"Project not found: {args.status}")
            return 1

        status = generate_status_json(project)
        if args.format == 'json':
            print(json.dumps(status, indent=2))
        else:
            print(f"\n{'='*60}")
            print(f"PROJECT STATUS: {project.title}")
            print(f"{'='*60}")
            print(f"ID: {project.project_id}")
            print(f"Status: {status['status']}")
            print(f"Current Phase: {status['phase']}")
            print(f"Progress: {status['progress']:.0%}")
            print(f"\nChapters: {status['chapters']}")
            print(f"Gaps: {status['gaps']}")
            if status.get('next_action'):
                print(f"\nNext Action: {status['next_action']}")
        return 0

    # Require outline or resume
    if not args.outline and not args.resume:
        parser.error("Must provide outline file or --resume PROJECT_ID")
        return 1

    # Parse output formats
    output_formats = [f.strip() for f in args.formats.split(',')]

    # Determine if compiling
    compile_output = args.compile or args.auto

    # Run workflow
    outline_path = Path(args.outline) if args.outline else None

    results = run_full_book_workflow(
        outline_path=outline_path,
        project_id=args.resume,
        auto=args.auto,
        phases=args.phases,
        compile_output=compile_output,
        output_formats=output_formats,
        tavily_budget=args.tavily_budget,
        approve_all_gaps=args.approve_all_gaps,
        approve_priority=args.approve_priority,
        dry_run=args.dry_run,
        verbose=args.verbose,
        evidence_first=args.evidence_first,
        gap_threshold=args.gap_threshold,
        writing_style=args.writing_style,
        writing_preset=args.writing_preset,
        writing_prompt_file=args.writing_prompt_file,
        reference_voice=args.reference_voice,
        polish_drafts=args.polish_drafts,
        delegate_drafting=args.delegate_drafting
    )

    # Output results
    if args.format == 'json':
        print(json.dumps(results, indent=2, default=str))
    elif not args.quiet:
        print(f"\n{'='*60}")
        if results.get('dry_run'):
            print("DRY RUN PREVIEW")
        elif results.get('success'):
            print("WORKFLOW COMPLETE")
        else:
            print("WORKFLOW FAILED")
        print(f"{'='*60}")

        print(f"Project: {results.get('project_id')}")
        print(f"Title: {results.get('title')}")

        if results.get('dry_run'):
            print(f"\nChapters: {results.get('chapters')}")
            print(f"Subjects: {len(results.get('subjects', []))}")
            print(f"Phases: {results.get('phases')}")
            print(f"\nEstimated Operations:")
            for op, count in results.get('estimated_operations', {}).items():
                print(f"  - {op}: {count}")
        else:
            if results.get('phases_run'):
                print(f"Phases Completed: {', '.join(map(str, results['phases_run']))}")

            if results.get('output_files'):
                print(f"\nOutput Files:")
                for fmt, path in results['output_files'].items():
                    print(f"  - {fmt.upper()}: {path}")

            if results.get('word_count'):
                print(f"\nWord Count: {results['word_count']:,}")

            if results.get('project_dir'):
                print(f"\nProject Directory: {results['project_dir']}")

        if not results.get('success') and not results.get('dry_run'):
            print(f"\nError: {results.get('error', 'Unknown error')}")

    return 0 if results.get('success', False) or results.get('dry_run') else 1


if __name__ == '__main__':
    sys.exit(main())
