# Developer Guide

Technical guide for extending and customizing the Research Development Framework.

---

## Architecture Overview

The framework is optimized for **Claude Code as the orchestrating agent**. RDF provides data operations while Claude Code handles planning, judgment, and synthesis.

```
┌─────────────────────────────────────────────────────────────────────────────┐
│                           CLAUDE CODE (Agent)                                │
│     Planning  │  Judgment  │  Synthesis  │  Quality Evaluation              │
└─────────────────────────────────────────────────────────────────────────────┘
                                    │
                                    ▼
┌─────────────────────────────────────────────────────────────────────────────┐
│                              RDF CLI LAYER                                   │
│   rdf write  │  rdf book  │  rdf essay  │  rdf research  │  rdf validate    │
└─────────────────────────────────────────────────────────────────────────────┘
                                    │
                                    ▼
┌─────────────────────────────────────────────────────────────────────────────┐
│                           DATA SUBSTRATE                                     │
│   Ingest/OCR  │  Chunk  │  Embed  │  Search  │  Quote Extract  │  Validate  │
└─────────────────────────────────────────────────────────────────────────────┘
                                    │
                                    ▼
┌─────────────────────────────────────────────────────────────────────────────┐
│                              STORAGE                                         │
│              PostgreSQL + pgvector  │  File System (artifacts)              │
└─────────────────────────────────────────────────────────────────────────────┘
```

**Key Principle:** RDF does data operations, Claude Code does thinking.

**Required Components:**
- Python 3.10+ (3.11 recommended)
- PostgreSQL 14+ with pgvector (16+ recommended)

---

## Project Layout

```
Research_development/
├── rdf                          # Unified CLI entrypoint
├── tools.json                   # Tool registry for agent orchestration
│
├── pipeline/                    # Core Python modules
│   ├── config.py               # Central configuration
│   ├── db_utils.py             # Database utilities + RAG functions
│   ├── cli_utils.py            # JSON response utilities
│   │
│   │  # Core Pipeline
│   ├── ingest_documents.py     # Document ingestion
│   ├── chunk_documents.py      # Text chunking + Semantic mode
│   ├── generate_embeddings.py  # Embedding generation
│   ├── extract_concepts.py     # Concept extraction
│   ├── search_export.py        # Search and export results
│   │
│   │  # Writing Workflows
│   ├── outline.py              # Book outline generator
│   ├── essay.py                # Essay workflow
│   ├── write.py                # Universal entry point
│   ├── book.py                 # Book workflow
│   ├── research_workflow.py    # 5-phase book research
│   ├── extract_quotes.py       # Quote extraction with citations
│   ├── polish_draft.py         # Draft style enhancement
│   ├── validate_draft.py       # Claim validation
│   │
│   │  # Utilities
│   ├── fetch.py                # Bounded chunk retrieval
│   ├── diff.py                 # File comparison
│   ├── assess.py               # Document assessment
│   ├── status.py               # Workflow state viewer
│   ├── provenance.py           # Minimal provenance tracking
│   │
│   │  # Library Management
│   ├── research_agent.py       # Autonomous research agent
│   ├── edit_metadata.py        # Metadata editing
│   ├── assess_quality.py       # OCR quality assessment
│   ├── reocr_document.py       # Re-OCR poor quality docs
│   ├── merge_concepts.py       # Concept deduplication
│   ├── library_gardener.py     # Library maintenance
│   ├── review_queue.py         # Governance queues
│   └── cite.py                 # Citation formatting
│
├── docs/                        # Documentation
├── config/
│   └── project.yaml            # Configuration
│
├── library/                     # Document storage
│   ├── ORGANIZED/              # Categorized documents
│   └── MARKDOWN_LIBRARY/       # Extracted markdown
│
├── projects/                    # Writing project outputs
│   ├── books/                   # Book projects
│   └── essays/                  # Essay projects
│   └── BOOK_xxx/               # Individual book folders
│       ├── outline.yaml        # Book outline
│       ├── workflow_state.json # Workflow state + resume tokens
│       ├── gaps.md             # Research gaps
│       └── chapter_*.md        # Generated chapters
│
├── essays/                      # Essay workflow outputs
│   └── ESSAY_xxx/              # Individual essay folders
│       ├── workflow_state.json # Workflow state
│       └── essay_polished.md   # Final essay
│
├── database/
│   └── schema.sql              # PostgreSQL schema
│
├── tests/                       # Test suite
├── .env                        # API keys
└── .env.db                     # Database credentials
```

---

## Tool Registry (tools.json)

`tools.json` is the **central registry** for Claude Code agent orchestration. It defines all available commands, their constraints, workflows, and error recovery patterns.

### Schema Overview

```json
{
  "version": "4.0.0",
  "description": "RDF Tool Registry - Agent Edition",

  "core_commands": { ... },       // Primary workflow commands
  "utility_commands": { ... },    // Support commands
  "workflow_constraints": { ... }, // Dependency rules
  "canonical_workflows": { ... },  // Standard workflow patterns
  "error_recovery": { ... },      // Error-to-command mappings
  "autonomy_levels": { ... }      // Checkpoint configurations
}
```

### Command Definitions

Each command entry maps a CLI command to its implementation:

```json
"rdf search": {
  "script": "pipeline/search_export.py",  // Python script
  "description": "Search the library",     // Human-readable
  "category": "retrieval",                 // Category for grouping
  "flags": {                               // Available flags
    "--summary": "Token-efficient output",
    "--limit": "Maximum results"
  },
  "default_flags": ["--strict-library-only"]  // Auto-applied flags
}
```

**Categories:**
| Category | Purpose |
|----------|---------|
| `library` | Document management (ingest, health, edit-meta) |
| `retrieval` | Search and fetch operations |
| `action` | Writing workflows (research, book, essay) |
| `quality` | Validation and polish |
| `governance` | Queues and status |
| `utility` | Misc tools (diff, assess) |

### Workflow Constraints

The `workflow_constraints` section defines **dependency rules** that the CLI enforces:

```json
"workflow_constraints": {
  "never_before": {
    "polish": ["validate"],        // polish requires validate first
    "draft": ["research", "quotes"],
    "book --phases 5": ["book --phases 1-4"]
  },
  "requires_checkpoint_approval": [
    "book --phases 5",
    "polish"
  ],
  "web_requires_explicit_approval": [
    "research --allow-web",
    "book --fill-gaps-from-web"
  ]
}
```

**Constraint Types:**

| Constraint | Effect |
|------------|--------|
| `never_before` | Command A cannot run until commands B have completed. Returns `QUEUE_GATE_BLOCKED` if violated. |
| `requires_checkpoint_approval` | Command pauses for human approval before executing |
| `web_requires_explicit_approval` | Web search requires explicit `--allow-web` flag |

**Error Recovery:** When a constraint is violated, the CLI returns an error code that maps to recovery commands in the `error_recovery` section.

### Canonical Workflows

Pre-defined workflow sequences for common operations:

```json
"canonical_workflows": {
  "essay": {
    "steps": ["search --summary", "research --strict", "quotes", "draft", "validate", "polish"],
    "checkpoints": ["post_search", "post_research", "post_quotes", "post_draft", "post_validation"],
    "default_autonomy": "full"
  },
  "book": {
    "steps": ["outline", "book --phases 1-4", "queue gap", "book --phases 5", "validate", "polish"],
    "checkpoints": ["post_outline", "post_research", "post_gap_resolution", "post_draft", "post_validation"],
    "default_autonomy": "supervised"
  }
}
```

These workflows are **informational** for the agent—they describe the expected sequence but don't enforce it. Actual enforcement is done by `workflow_constraints`.

### Error Recovery Mappings

Maps error codes to suggested recovery commands:

```json
"error_recovery": {
  "CORRUPT_PDF": "rdf ingest {path} --ocr-profile standard",
  "INSUFFICIENT_RESEARCH": "rdf research \"{query}\" --allow-web",
  "GAP_THRESHOLD_NOT_MET": "rdf queue list gap",
  "VALIDATION_FAILED": "rdf queue list validation",
  "QUEUE_GATE_BLOCKED": "rdf queue list"
}
```

When a command fails with one of these codes, the agent can use the suggested command to resolve the issue.

### Configuration Checks

Some features require explicit user configuration before use. The `configuration_checks` section defines these:

```json
"configuration_checks": {
  "entity_extraction": {
    "check_command": "rdf config check --format json",
    "configure_command": "rdf config entity-extraction --provider {provider}",
    "required_for": ["rdf graph", "concept extraction during ingest"],
    "options": ["gliner", "openai", "hybrid"],
    "recommended": "hybrid"
  }
}
```

**Handling CONFIGURATION_REQUIRED responses:**

When a command returns `CONFIGURATION_REQUIRED`, the response includes a `decision_packet`:

```json
{
  "status": "error",
  "code": "CONFIGURATION_REQUIRED",
  "message": "Entity extraction has not been configured.",
  "actionable_advice": "Run: rdf config entity-extraction",
  "decision_packet": {
    "decision_id": "entity_extraction_provider",
    "question": "Which entity extraction provider should be used?",
    "options": [
      {"id": "gliner", "label": "GLiNER (Fast/Free)", "description": "CPU-based, no API costs"},
      {"id": "openai", "label": "OpenAI (Quality)", "description": "Best quality, ~$0.01/1K tokens"},
      {"id": "hybrid", "label": "Hybrid (Recommended)", "description": "GLiNER + OpenAI relations"}
    ],
    "default": "hybrid"
  }
}
```

Claude Code should:
1. Present the options to the user
2. Collect their choice
3. Run the configure command: `rdf config entity-extraction --provider <choice>`
4. Retry the original command

### Autonomy Levels

Defines checkpoint behavior for each autonomy level:

```json
"autonomy_levels": {
  "full": {
    "description": "Run to completion, no pauses",
    "checkpoints_enabled": false
  },
  "supervised": {
    "description": "Pause after research and draft",
    "checkpoints_enabled": ["post_research", "post_draft"]
  },
  "interactive": {
    "description": "Pause at all checkpoints",
    "checkpoints_enabled": true
  }
}
```

### Updating tools.json

When adding a new CLI command:

1. **Add to rdf script:** Register in `COMMAND_MAP` in the `rdf` entrypoint
2. **Add to tools.json:** Add entry to `core_commands` or `utility_commands`
3. **Update constraints:** If the command has dependencies, add to `workflow_constraints`
4. **Update workflows:** If the command is part of a standard workflow, update `canonical_workflows`
5. **Add error codes:** If the command can produce unique errors, add to `error_recovery`

**Example: Adding a new "rdf summarize" command:**

```json
// In core_commands
"rdf summarize": {
  "script": "pipeline/summarize.py",
  "description": "Summarize a document or chunk",
  "category": "retrieval",
  "flags": {
    "--max-length": "Maximum summary length",
    "--style": "Summary style (bullets, prose)"
  }
}
```

---

## Configuration System

### Environment Files

**`.env`** - API and application settings:
```bash
OPENAI_API_KEY=your_key
EMBEDDING_MODEL=text-embedding-3-small
EMBEDDING_DIMENSIONS=1536
LOG_LEVEL=INFO
```

**`.env.db`** - Database credentials:
```bash
DB_HOST=localhost
DB_PORT=5432
DB_NAME=research_dev_db
DB_USER=research_dev_user
DB_PASSWORD=your_secure_password
```

### Project Configuration

`config/project.yaml` uses a simplified structure:

```yaml
# Database (uses .env.db by default)
database:
  url: ${DATABASE_URL}

# Embeddings configuration
embeddings:
  provider: "openai"              # "openai" or "local"
  model: "text-embedding-3-small"
  dimension: 1536

# Agent safety defaults
agent_safety:
  allow_web_search: false         # Library only by default
  require_approved_web_sources: true
  validation_required_before_polish: true

# Workflow defaults
defaults:
  autonomy_essay: "full"
  autonomy_book: "supervised"
  strict_library_only: true

# Search settings
search:
  default_mode: "hybrid"
  max_results: 100
  default_limit: 10
```

### Configuration Module

`pipeline/config.py` centralizes all settings:

```python
from config import (
    DB_CONFIG,
    OPENAI_API_KEY,
    EMBEDDING_MODEL,
    CHUNK_CONFIG,
    PATHS,
    PROCESSING_CONFIG,
    QUALITY_THRESHOLDS
)
```

---

## Database Utilities

### Connection Management

```python
from db_utils import get_db_connection, get_dict_cursor

# Using context manager
with get_db_connection() as conn:
    with conn.cursor() as cur:
        cur.execute("SELECT * FROM documents")
        results = cur.fetchall()

# Using dict cursor
with get_dict_cursor() as (conn, cur):
    cur.execute("SELECT * FROM documents WHERE document_id = %s", (doc_id,))
    row = cur.fetchone()  # Returns dict
```

### Common Operations

```python
from db_utils import (
    # Basic operations
    insert_document,
    insert_chunk,
    batch_insert_chunks,
    update_embedding,
    batch_update_embeddings,
    update_document_status,
    get_document,
    get_chunks_for_document,
    semantic_search,
    keyword_search,
    get_database_stats,

    # Advanced RAG
    rerank_results,              # Cross-encoder re-ranking
    hybrid_search_with_rerank,   # Combined search with RRF + re-ranking
    graphrag_search,             # Knowledge graph traversal
    get_related_concepts,        # Multi-hop concept lookup
    get_chunk_with_context,      # Context-aware retrieval
)

# Insert document
doc_data = {
    'document_id': 'DOC_001',
    'title': 'Example Document',
    'word_count': 5000,
    'processing_status': 'ingested'
}
insert_document(doc_data)

# Batch insert chunks
chunks = [
    {'chunk_id': 'DOC_001_C0001', 'document_id': 'DOC_001', ...},
    {'chunk_id': 'DOC_001_C0002', 'document_id': 'DOC_001', ...},
]
batch_insert_chunks(chunks)

# Search
results = semantic_search(query_embedding, limit=10)
results = keyword_search("search term", limit=10)
```

---

## Provenance API

The framework uses minimal provenance with 6 essential fields:

```python
from provenance import MinimalProvenance, UsedInTrace

# Create provenance for a chunk
prov = MinimalProvenance(
    doc_id="DOC_023",
    source_type="library",  # or "web"
    page_range=[15, 17],
    chunk_id="DOC_023_C0045",
    extraction="standard",  # or "ocr", "academic"
    content_hash="sha256:abc123..."
)

# Convert to dict for JSON serialization
prov_dict = prov.to_dict()

# Track usage in output artifacts
trace = UsedInTrace(
    output_artifact="projects/books/BOOK_001/chapter_01.md",
    provenance_list=[prov1, prov2, prov3]
)
```

### Web Source Provenance

```python
from provenance import WebSourceProvenance

web_prov = WebSourceProvenance(
    url="https://example.com/article",
    retrieved_at="2025-12-12T10:30:00Z",
    search_query="original query",
    approval_status="approved",  # pending, approved, rejected
    approval_queue_id="web_001"
)
```

---

## CLI Response Format

All CLI commands return structured JSON with `--format json`:

### Standard Response Wrapper

```python
from cli_utils import success_response, error_response, pause_response

# Success response
response = success_response(
    message="Operation completed",
    data={"results": [...]},
    code="SUCCESS"
)

# Error response
response = error_response(
    message="Coverage 40% below threshold 50%",
    code="GAP_THRESHOLD_NOT_MET",
    data={"coverage": 40, "threshold": 50},
    actionable_advice="Run rdf research on missing topics."
)

# Pause for review
response = pause_response(
    message="Research complete. Review gaps before drafting.",
    checkpoint="post_research",
    workflow_id="BOOK_xxx",
    resume_token="RESUME_xxx_post_research",
    review_artifacts=[
        {"path": "projects/books/BOOK_xxx/gaps.md", "description": "Gaps requiring strategy"}
    ],
    decision_packet={
        "decision_id": "gap_fill_strategy",
        "question": "How should gaps be handled?",
        "options": [
            {"id": "library_only", "label": "Library only"},
            {"id": "allow_web", "label": "Allow web"},
            {"id": "skip", "label": "Skip gaps"}
        ],
        "default": "library_only"
    }
)
```

### Response Codes

| Code | Category | Meaning |
|------|----------|---------|
| `SUCCESS` | Success | Operation completed normally |
| `PAUSED_FOR_REVIEW` | Success | Waiting for human decision |
| `RESULTS_FOUND` | Success | Search returned results |
| `NO_RESULTS` | Success | Search completed but empty |
| `GAP_THRESHOLD_NOT_MET` | Error | Insufficient research coverage |
| `CORRUPT_PDF` | Error | Cannot extract from document |
| `VALIDATION_FAILED` | Error | Claims not supported |
| `INSUFFICIENT_RESEARCH` | Error | Not enough sources found |

---

## Advanced RAG APIs

### Cross-Encoder Re-ranking

Re-rank search results using precise relevance scoring:

```python
from db_utils import rerank_results, hybrid_search_with_rerank

# Option 1: Re-rank existing results
results = semantic_search(query_embedding, limit=50)
reranked = rerank_results(
    query="What influences the etheric body?",
    results=results,
    top_k=10,
    text_field='chunk_text'
)

# Option 2: Combined hybrid search with re-ranking (recommended)
results = hybrid_search_with_rerank(
    query_text="What influences the etheric body?",
    limit=10,              # Final results
    initial_fetch=50,      # Candidates for re-ranking
    use_rerank=True        # Enable cross-encoder
)
```

### GraphRAG Retrieval

Use knowledge graph traversal for multi-hop reasoning:

```python
from db_utils import (
    graphrag_search,
    find_concept_by_name,
    get_related_concepts,
    get_chunks_for_concepts
)

# Full GraphRAG search
result = graphrag_search(
    query="What influences the etheric body?",
    limit=10,
    hop_depth=2,                # Relationship hops
    min_cooccurrence=2,         # Min co-occurrence strength
    include_direct_search=True  # Also include hybrid search
)

print(f"Concepts found: {result['concepts']}")
print(f"Graph paths: {result['graph_paths']}")
print(f"Results: {len(result['results'])} chunks")
```

### Semantic Chunking

Use embedding-based boundary detection:

```python
from chunk_documents import SemanticChunker

chunker = SemanticChunker(
    min_tokens=100,
    max_tokens=1000,
    target_tokens=500,
    similarity_threshold=0.5,  # Lower = more sensitive
    window_size=3              # Sentences per comparison window
)

chunks = chunker.chunk_semantic(text, document_id)
```

---

## Adding New Document Formats

### Step 1: Create Extractor

Add a new extractor method in `ingest_documents.py`:

```python
def extract_text_from_newformat(self, file_path: Path) -> Tuple[str, Dict]:
    """Extract text from new format."""
    from newformat_lib import read_newformat

    content = read_newformat(file_path)
    text = content.get_text()

    metadata = {
        'title': content.title,
        'author': content.author,
        'page_count': len(content.pages)
    }

    return text, metadata
```

### Step 2: Register Extractor

Add to the extractors dictionary in `process_file()`:

```python
extractors = {
    '.pdf': self.extract_text_from_pdf,
    '.docx': self.extract_text_from_docx,
    '.txt': self.extract_text_from_txt,
    '.md': self.extract_text_from_markdown,
    '.newformat': self.extract_text_from_newformat,  # Add this
}
```

### Step 3: Update Configuration

Add to `PROCESSING_CONFIG` in `config.py`:

```python
PROCESSING_CONFIG = {
    'supported_extensions': ['.pdf', '.docx', '.txt', '.md', '.newformat'],
    ...
}
```

---

## Customizing Chunking

### Modify Chunk Size

In `config.py`:

```python
CHUNK_CONFIG = {
    'target_tokens': 750,    # Increase/decrease as needed
    'overlap_tokens': 100,   # More overlap = more context preservation
    'min_tokens': 100,       # Minimum chunk size
    'max_tokens': 1000,      # Maximum chunk size
    'encoding': 'cl100k_base',
}
```

### Custom Chunking Strategy

Override `chunk_text()` in `chunk_documents.py`:

```python
def chunk_by_chapters(self, text: str, document_id: str):
    """Chunk document by chapter boundaries."""
    chapters = re.split(r'\n#+ Chapter \d+', text)

    for i, chapter in enumerate(chapters):
        if len(chapter.strip()) < 100:
            continue

        yield {
            'chunk_id': f"{document_id}_CH{i:02d}",
            'document_id': document_id,
            'chunk_sequence': i,
            'chunk_text': chapter.strip(),
            'chunk_tokens': self.count_tokens(chapter)
        }
```

---

## Adding New Pipeline Commands

### Command Template

Create new commands in `pipeline/`:

```python
#!/usr/bin/env python3
"""
Custom Tool - Description of what it does.
"""
import argparse
import json
import logging
from cli_utils import success_response, error_response
from db_utils import get_db_connection

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

def main():
    parser = argparse.ArgumentParser(description="Custom tool")
    parser.add_argument('--format', choices=['text', 'json'], default='text')
    parser.add_argument('--verbose', '-v', action='store_true')
    args = parser.parse_args()

    try:
        # Your logic here
        result = do_something()

        if args.format == 'json':
            response = success_response(
                message="Operation completed",
                data=result
            )
            print(json.dumps(response, indent=2))
        else:
            print(f"Result: {result}")

    except Exception as e:
        if args.format == 'json':
            response = error_response(
                message=str(e),
                code="CUSTOM_ERROR"
            )
            print(json.dumps(response, indent=2))
        else:
            logger.error(f"Error: {e}")
        raise SystemExit(1)

if __name__ == '__main__':
    main()
```

### Register in rdf CLI

Add to the `COMMAND_MAP` in the `rdf` script:

```python
COMMAND_MAP = {
    # ... existing commands ...
    "custom": ("custom_tool.py", "Description of custom tool"),
}
```

---

## Extending the Schema

### Add New Table

1. Create migration SQL:

```sql
-- migrations/005_custom_table.sql
CREATE TABLE IF NOT EXISTS custom_table (
    id SERIAL PRIMARY KEY,
    document_id VARCHAR(100) REFERENCES documents(document_id),
    custom_field VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX idx_custom_document ON custom_table(document_id);
```

2. Apply migration:

```bash
PGPASSWORD='...' psql -h localhost -U research_dev_user -d research_dev_db \
    -f migrations/005_custom_table.sql
```

3. Add to `schema.sql` for future deployments.

---

## Testing

### Running Tests

```bash
cd tests

# Quick smoke tests
python3 run_tests.py --quick

# Full test suite
python3 run_tests.py
```

### Unit Test Example

```python
# tests/test_db_utils.py
import pytest
from pipeline.db_utils import get_database_stats, execute_query

def test_database_connection():
    """Test database connectivity."""
    stats = get_database_stats()
    assert stats is not None
    assert 'total_documents' in stats

def test_execute_query():
    """Test query execution."""
    result = execute_query("SELECT 1 as value", fetch='one')
    assert result['value'] == 1
```

---

## Logging

### Configure Logging

In `config.py`:

```python
LOGGING_CONFIG = {
    'level': 'DEBUG',  # DEBUG, INFO, WARNING, ERROR
    'format': '%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    'file': BASE_DIR / 'logs' / 'pipeline.log',
}
```

### Use Logging

```python
import logging
from config import LOGGING_CONFIG

logging.basicConfig(
    level=getattr(logging, LOGGING_CONFIG['level']),
    format=LOGGING_CONFIG['format']
)
logger = logging.getLogger(__name__)

logger.info("Processing started")
logger.warning("Low quality document detected")
logger.error(f"Processing failed: {error}")
```

---

## Performance Optimization

### Database Indexes

The schema includes optimized indexes, but consider:

```sql
-- For high-volume chunk queries
CREATE INDEX CONCURRENTLY idx_chunks_doc_seq
    ON chunks(document_id, chunk_sequence);

-- For frequent concept lookups
CREATE INDEX CONCURRENTLY idx_concepts_name_lower
    ON concepts(LOWER(name));
```

### Embedding Index Tuning

For large datasets, tune the IVFFlat index:

```sql
-- Drop and recreate with more lists for large datasets
DROP INDEX idx_chunks_embedding;
CREATE INDEX idx_chunks_embedding ON chunks
    USING ivfflat (embedding vector_cosine_ops)
    WITH (lists = 200);  -- Increase for larger datasets
```

---

## Troubleshooting

### Common Issues

**Database Connection Refused:**
```bash
# Check PostgreSQL is running
sudo systemctl status postgresql

# Check connection settings
psql -h localhost -U research_dev_user -d research_dev_db
```

**OpenAI API Errors:**
```bash
# Test API key
python -c "from openai import OpenAI; print(OpenAI().models.list())"
```

**Import Errors:**
```bash
# Ensure dependencies installed
pip install -r requirements.txt

# Check Python path
python -c "import sys; print(sys.path)"
```

### Debug Mode

Enable verbose logging:

```python
import logging
logging.getLogger().setLevel(logging.DEBUG)
```

Or set environment variable:

```bash
export LOG_LEVEL=DEBUG
```

---

## Next Steps

- [Setup Guide](SETUP.md) - Installation instructions
- [CLI User Guide](CLI_USER_GUIDE.md) - Complete CLI reference
- [Canonical Workflows](CANONICAL_WORKFLOWS.md) - Standard workflows
- [Database Schema](DATABASE_SCHEMA.md) - PostgreSQL schema reference
