# Canonical Workflows

The 5 core workflows for daily use with the Research Development Framework. While 17 commands are available, these 5 workflows cover 95% of use cases.

---

## Quick Reference

| Workflow | Purpose | Primary Command |
|----------|---------|-----------------|
| **1. Daily Ingest** | Add new documents | `./rdf ingest NEW_DOCS/` |
| **2. Research Session** | Answer questions | `./rdf research "query" --strict` |
| **3. Quote Bank** | Gather evidence | `./rdf quotes "topic" --format json` |
| **4. Evidence-First Draft** | Write grounded content | `./rdf essay "topic" --evidence-first` |
| **5. Validate & Polish** | Quality assurance | `./rdf validate` + `./rdf polish` |

---

## Workflow 1: Daily Ingest

**Purpose:** Add new documents to your library with full processing.

### Quick Version
```bash
# Drop files and process
cp ~/Downloads/*.pdf NEW_DOCS/
./rdf ingest NEW_DOCS/
```

### With Options
```bash
# Force re-processing of existing documents
./rdf ingest NEW_DOCS/ --force

# Use OCR for scanned documents
./rdf ingest old_scan.pdf --ocr-profile standard

# JSON output for automation
./rdf ingest NEW_DOCS/ --format json
```

### Quality Check
```bash
# Check library health after ingestion
./rdf health --format json
```

### When to Use
- Daily/weekly document additions
- Building your research library
- Adding new sources for a project

---

## Workflow 2: Research Session

**Purpose:** Answer complex research questions using your library.

### Quick Version
```bash
./rdf research "What influenced Steiner's epistemology?" --strict -o report.md
```

### With Options
```bash
# Library only (default, recommended)
./rdf research "query" --strict

# Allow web gap-filling (requires approval)
./rdf research "query" --allow-web --max-iterations 5

# With budget limit
./rdf research "query" --budget 1.00

# JSON output for programmatic use
./rdf research "query" --format json
```

### For Quick Searches
```bash
# Token-efficient preview (titles only)
./rdf search "query" --summary

# Full results with limit
./rdf search "query" --mode hybrid --limit 20

# Semantic search only
./rdf search "query" --mode semantic --limit 10
```

### When to Use
- Exploring a topic
- Answering specific questions
- Finding sources for a claim
- Understanding connections between concepts

---

## Workflow 3: Quote Bank (Chapter Preparation)

**Purpose:** Gather relevant quotes with citations for a chapter or topic.

### Quick Version
```bash
./rdf quotes "consciousness and thinking" --format json
```

### With Options
```bash
# Limit number of quotes
./rdf quotes "topic" --max 30

# Minimum quote length
./rdf quotes "topic" --min-length 100

# JSON output for processing
./rdf quotes "alchemy symbols" --format json
```

### For Multiple Topics
```bash
# Build quote banks for each chapter subject
./rdf quotes "mystery schools" --format json > ch1_quotes.json
./rdf quotes "alchemy" --format json > ch2_quotes.json
./rdf quotes "kabbalah" --format json > ch3_quotes.json
```

**Output:** Creates `quote_bank.json` with citations.

### When to Use
- Preparing to write a chapter
- Building evidence for an argument
- Collecting sources for a literature review
- Pre-work for evidence-first drafting

---

## Workflow 4: Evidence-First Draft

**Purpose:** Generate essays or book chapters grounded in your sources (prevents hallucination).

### Essay (Quick Version)
```bash
# Single-command essay generation
./rdf essay "Freemasonry in the 1800s" --pages 20 --evidence-first

# Output: essays/ESSAY_xxx/essay_polished.md
```

### Book (Full Workflow)
```bash
# 1. Generate outline from your idea
./rdf outline "History of Western Esotericism" --chapters 5 --pages 100

# 2. Run book workflow (supervised mode pauses for review)
./rdf book projects/books/BOOK_xxx/outline.yaml --autonomy supervised

# 3. Review gaps.md at checkpoint
cat projects/books/BOOK_xxx/gaps.md

# 4. Resume after review
./rdf book --resume BOOK_xxx --phases 5

# 5. Find your chapters
ls projects/books/BOOK_xxx/
# → chapter_01.md, chapter_02.md, ...
```

### Autonomy Levels

| Level | Behavior | Best For |
|-------|----------|----------|
| `--autonomy full` | No pauses, runs to completion | "Just write it" |
| `--autonomy supervised` | Pauses after research and draft | Balanced oversight (book default) |
| `--autonomy interactive` | Pauses at all checkpoints | Careful, iterative work |

### Writing Style Options
```bash
# Preset styles
./rdf essay "topic" --style academic
./rdf essay "topic" --style accessible
./rdf essay "topic" --style narrative
./rdf essay "topic" --style popular

# For books
./rdf book outline.yaml --style academic
```

### When to Use
- Writing book chapters
- Creating research reports
- Generating content that must be grounded in sources
- Any writing where hallucination is unacceptable

---

## Workflow 5: Validate & Polish

**Purpose:** Verify claims and enhance writing quality.

### Quick Version
```bash
# Validate claims against library
./rdf validate chapter.md

# Polish with citation preservation
./rdf polish chapter.md --preset accessible
```

### Full Quality Assurance
```bash
# 1. Validate claims
./rdf validate chapter.md --format json

# 2. Review validation issues
./rdf queue list validation

# 3. Approve or reject findings
./rdf queue approve val_001
./rdf queue reject val_002 --reason hallucination --note "Entity not found"

# 4. Polish after fixes
./rdf polish chapter.md --preset accessible

# 5. Final validation
./rdf validate chapter_polished.md
```

### Validation Categories
- **Supported**: Claim has source evidence
- **Unverified**: No matching sources found (research needed)
- **Contradicted**: Sources contradict claim (fix required)
- **Misattributed**: Wrong source attribution (correction provided)

### Polish Presets
- `academic` - Formal, scholarly tone
- `accessible` - Clear, general audience
- `narrative` - Story-like flow
- `concise` - Tighter, shorter prose
- `popular` - Engaging, magazine style

### When to Use
- After any draft generation
- Before publishing or submitting
- Quality assurance for AI-generated content
- Fact-checking your own writing

---

## Putting It Together: Complete Book Project

```bash
# Day 1: Ingest sources
cp research_pdfs/*.pdf NEW_DOCS/
./rdf ingest NEW_DOCS/

# Day 2: Research and quote gathering
./rdf research "overview of topic" --strict -o research_notes.md
./rdf quotes "chapter 1 topic" --format json > ch1_quotes.json
./rdf quotes "chapter 2 topic" --format json > ch2_quotes.json

# Day 3: Generate outline and drafts
./rdf outline "My Book Title" --chapters 5 --pages 100
./rdf book projects/books/BOOK_xxx/outline.yaml --autonomy supervised

# Review gaps.md at checkpoint, then resume:
./rdf book --resume BOOK_xxx --phases 5

# Day 4+: Review, validate, polish
# HUMAN REVIEW of drafts
./rdf validate projects/books/BOOK_xxx/chapter_01.md
./rdf queue list validation
./rdf queue approve val_001
./rdf polish projects/books/BOOK_xxx/chapter_01.md --preset narrative
```

---

## Quick Essay (Minimal Steps)

```bash
# One command for a complete essay
./rdf essay "The evolution of consciousness in esoteric thought" --pages 15 --style academic

# Find your essay
ls essays/ESSAY_xxx/
# → essay_polished.md
```

---

## Utility Commands

These commands support the main workflows:

| Command | Purpose | Example |
|---------|---------|---------|
| `rdf status` | View workflow state | `./rdf status` |
| `rdf fetch` | Retrieve specific chunk | `./rdf fetch chunk_023_045` |
| `rdf diff` | Compare files | `./rdf diff draft.md polished.md` |
| `rdf assess` | Document metadata | `./rdf assess DOC_023` |
| `rdf health` | Library health scan | `./rdf health --format json` |
| `rdf edit-meta` | Fix metadata | `./rdf edit-meta DOC_023 --title "Correct"` |
| `rdf graph` | Query knowledge graph | `./rdf graph query "consciousness"` |

---

## Troubleshooting Quick Reference

| Problem | Solution |
|---------|----------|
| Poor search results | Try `--mode hybrid` or regenerate embeddings |
| Hallucinated content | Use `--evidence-first` mode |
| Missing sources | Run `rdf research` with gap query |
| OCR quality issues | Run `rdf ingest --ocr-profile standard` |
| Workflow stuck | Check `rdf status` and resume with `--resume` |

---

## Session Resumption

When resuming work on an existing project (e.g., after starting a new Claude Code session):

### Quick Resume
```bash
# Get context packet for warm-start
./rdf context BOOK_xxx --format json

# Returns: phase, progress, blocking items, next actions
```

### Full Resume Workflow
```bash
# 1. Load project context
./rdf context BOOK_xxx

# 2. Check for blocking items in queue
./rdf queue list --project BOOK_xxx

# 3. Resolve any blocking items
./rdf queue approve gap_001
./rdf queue reject gap_002 --reason needs_web --feedback "Use academic sources only"

# 4. Continue workflow
./rdf book projects/books/BOOK_xxx/outline.yaml --resume
```

### What `rdf context` Returns
- Current phase and progress percentage
- Active chapter and subject
- Recent activity log
- Pending decisions and blocking items
- Resource status (API credits, budget)
- Suggested next actions

---

## Bibliography Export

After completing a project, export the bibliography for reference managers:

```bash
# Export as BibTeX (for LaTeX)
./rdf export bibliography --project BOOK_xxx --format bibtex --output refs.bib

# Export as RIS (for Zotero, Mendeley, EndNote)
./rdf export bibliography --project BOOK_xxx --format ris --output refs.ris

# Export as CSL-JSON (for Pandoc)
./rdf export bibliography --project BOOK_xxx --format csl-json --output refs.json

# Validate sources first (check for missing fields)
./rdf export bibliography --project BOOK_xxx --validate
```

---

## Next Steps

- **New user?** Start with Workflow 1 (Daily Ingest) and Workflow 2 (Research Session)
- **Writing a book?** Use Workflow 4 (Evidence-First Draft) with Workflow 5 (Validate & Polish)
- **Building a library?** Focus on Workflow 1 with periodic `rdf health` checks

For complete CLI documentation, see [CLI_USER_GUIDE.md](CLI_USER_GUIDE.md).
