-- =============================================================================
-- MIGRATION 001: Writer's Workbench Tables
-- =============================================================================
-- Adds tables for the Writer's Studio functionality:
--   - Research pins (the missing link between search and writing)
--   - Chat history (persistent conversations)
--   - Concept aliases (for taxonomy cleanup)
--
-- Run with:
--   PGPASSWORD='JpGZhjgjNd1M8rrh29BT' psql -h localhost -U research_dev_user \
--     -d research_dev_db -f database/migrations/001_writer_workbench.sql
-- =============================================================================

-- -----------------------------------------------------------------------------
-- Research Pins Table (The Missing Link)
-- -----------------------------------------------------------------------------
-- Allows users to "pin" chunks/documents to projects while researching.
-- This bridges the gap between finding information and using it in writing.
-- -----------------------------------------------------------------------------
CREATE TABLE IF NOT EXISTS research_pins (
    pin_id SERIAL PRIMARY KEY,

    -- What is pinned (either a chunk or whole document)
    chunk_id VARCHAR(100) REFERENCES chunks(chunk_id) ON DELETE CASCADE,
    document_id VARCHAR(100) REFERENCES documents(document_id) ON DELETE CASCADE,

    -- Which project it belongs to (NULL = general pinboard)
    project_id INTEGER REFERENCES book_projects(project_id) ON DELETE SET NULL,

    -- User annotation
    pin_title VARCHAR(255),              -- Short label for the pin
    pin_note TEXT,                        -- User's notes about why this is relevant
    highlight_text TEXT,                  -- Specific text selection within the chunk

    -- Organization
    pin_color VARCHAR(20) DEFAULT 'default',  -- For visual categorization
    pin_tags TEXT[],                      -- Array of user-defined tags
    sort_order INTEGER DEFAULT 0,         -- For manual ordering

    -- Source context (cached for quick display)
    source_title VARCHAR(500),            -- Document title (cached)
    source_author VARCHAR(255),           -- Author name (cached)
    source_page INTEGER,                  -- Page number if known

    -- Metadata
    is_used BOOLEAN DEFAULT FALSE,        -- Has this been inserted into a chapter?
    used_in_chapter_id INTEGER REFERENCES book_chapters(chapter_id),

    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Indexes for efficient queries
CREATE INDEX IF NOT EXISTS idx_pins_project ON research_pins(project_id);
CREATE INDEX IF NOT EXISTS idx_pins_document ON research_pins(document_id);
CREATE INDEX IF NOT EXISTS idx_pins_chunk ON research_pins(chunk_id);
CREATE INDEX IF NOT EXISTS idx_pins_created ON research_pins(created_at DESC);
CREATE INDEX IF NOT EXISTS idx_pins_tags ON research_pins USING gin(pin_tags);

-- Trigger for updated_at
DROP TRIGGER IF EXISTS tr_pins_updated_at ON research_pins;
CREATE TRIGGER tr_pins_updated_at BEFORE UPDATE ON research_pins
    FOR EACH ROW EXECUTE FUNCTION update_updated_at();

-- -----------------------------------------------------------------------------
-- Chat History Table
-- -----------------------------------------------------------------------------
-- Persists RAG chat conversations for review and reference.
-- Allows users to return to previous research conversations.
-- -----------------------------------------------------------------------------
CREATE TABLE IF NOT EXISTS chat_history (
    chat_id SERIAL PRIMARY KEY,

    -- Session management
    session_id VARCHAR(100) NOT NULL,     -- Groups messages into conversations
    session_title VARCHAR(255),           -- User-editable conversation title

    -- The conversation
    role VARCHAR(20) NOT NULL,            -- 'user' or 'assistant'
    message TEXT NOT NULL,                -- The actual message content

    -- For assistant messages: what sources were used
    sources_used JSONB,                   -- Array of {chunk_id, document_id, relevance}
    intelligence_mode VARCHAR(50),        -- cloud, local, statistical
    model_used VARCHAR(100),              -- gpt-4o, llama3, extractive, etc.

    -- Context filters that were active
    context_filters JSONB,                -- {document_id, author, topic, etc.}

    -- Metadata
    tokens_used INTEGER,                  -- For cost tracking (cloud mode)
    response_time_ms INTEGER,             -- Performance tracking

    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Indexes
CREATE INDEX IF NOT EXISTS idx_chat_session ON chat_history(session_id);
CREATE INDEX IF NOT EXISTS idx_chat_created ON chat_history(created_at DESC);
CREATE INDEX IF NOT EXISTS idx_chat_role ON chat_history(session_id, role);

-- -----------------------------------------------------------------------------
-- Chat Sessions Table
-- -----------------------------------------------------------------------------
-- Manages chat conversation sessions (metadata level)
-- -----------------------------------------------------------------------------
CREATE TABLE IF NOT EXISTS chat_sessions (
    session_id VARCHAR(100) PRIMARY KEY,

    title VARCHAR(255),                   -- Auto-generated or user-edited
    description TEXT,

    -- Link to project if researching for specific book
    project_id INTEGER REFERENCES book_projects(project_id) ON DELETE SET NULL,

    -- Context that applies to whole session
    context_document_id VARCHAR(100) REFERENCES documents(document_id),
    context_filters JSONB,

    -- Stats
    message_count INTEGER DEFAULT 0,

    -- Metadata
    is_starred BOOLEAN DEFAULT FALSE,     -- Mark important conversations
    is_archived BOOLEAN DEFAULT FALSE,    -- Hide from active list

    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_message_at TIMESTAMP
);

CREATE INDEX IF NOT EXISTS idx_sessions_project ON chat_sessions(project_id);
CREATE INDEX IF NOT EXISTS idx_sessions_starred ON chat_sessions(is_starred) WHERE is_starred = TRUE;
CREATE INDEX IF NOT EXISTS idx_sessions_active ON chat_sessions(is_archived, last_message_at DESC);

-- Trigger for updated_at
DROP TRIGGER IF EXISTS tr_sessions_updated_at ON chat_sessions;
CREATE TRIGGER tr_sessions_updated_at BEFORE UPDATE ON chat_sessions
    FOR EACH ROW EXECUTE FUNCTION update_updated_at();

-- -----------------------------------------------------------------------------
-- Concept Aliases Table (Taxonomy Gardener)
-- -----------------------------------------------------------------------------
-- Allows merging duplicate concepts without losing data.
-- When concept A is aliased to concept B, searches for A also find B.
-- -----------------------------------------------------------------------------
CREATE TABLE IF NOT EXISTS concept_aliases (
    alias_id SERIAL PRIMARY KEY,

    -- The alias term (e.g., "Etheric Forces")
    alias_name VARCHAR(100) NOT NULL,
    alias_name_normalized VARCHAR(100),   -- Lowercase for matching

    -- The canonical concept it maps to (e.g., "Etheric Body")
    canonical_concept_id INTEGER REFERENCES concepts(concept_id) ON DELETE CASCADE,

    -- Why this alias exists
    alias_type VARCHAR(50) DEFAULT 'synonym',  -- synonym, alternate_spelling, translation
    notes TEXT,

    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE INDEX IF NOT EXISTS idx_aliases_name ON concept_aliases(alias_name_normalized);
CREATE INDEX IF NOT EXISTS idx_aliases_canonical ON concept_aliases(canonical_concept_id);
CREATE UNIQUE INDEX IF NOT EXISTS idx_aliases_unique ON concept_aliases(alias_name_normalized, canonical_concept_id);

-- -----------------------------------------------------------------------------
-- User Preferences Table
-- -----------------------------------------------------------------------------
-- Stores per-user settings (for future multi-user support)
-- For now, single user with user_id = 'default'
-- -----------------------------------------------------------------------------
CREATE TABLE IF NOT EXISTS user_preferences (
    pref_id SERIAL PRIMARY KEY,
    user_id VARCHAR(100) DEFAULT 'default',

    -- UI Preferences
    theme VARCHAR(50) DEFAULT 'light',
    default_search_mode VARCHAR(50) DEFAULT 'hybrid',
    results_per_page INTEGER DEFAULT 20,

    -- Citation preferences
    citation_format VARCHAR(50) DEFAULT 'chicago',  -- chicago, apa, mla, harvard

    -- Default project for quick pins
    default_project_id INTEGER REFERENCES book_projects(project_id),

    -- Feature flags
    show_source_preview BOOLEAN DEFAULT TRUE,
    auto_save_chat BOOLEAN DEFAULT TRUE,

    -- Cached recent items
    recent_documents JSONB,               -- Array of recently viewed doc IDs
    recent_searches JSONB,                -- Array of recent search queries

    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    UNIQUE(user_id)
);

-- Insert default user
INSERT INTO user_preferences (user_id)
VALUES ('default')
ON CONFLICT (user_id) DO NOTHING;

-- Trigger for updated_at
DROP TRIGGER IF EXISTS tr_prefs_updated_at ON user_preferences;
CREATE TRIGGER tr_prefs_updated_at BEFORE UPDATE ON user_preferences
    FOR EACH ROW EXECUTE FUNCTION update_updated_at();

-- -----------------------------------------------------------------------------
-- Helpful Views
-- -----------------------------------------------------------------------------

-- View: Pins with full context
CREATE OR REPLACE VIEW v_pins_full AS
SELECT
    p.pin_id,
    p.project_id,
    bp.project_name,
    p.chunk_id,
    p.document_id,
    COALESCE(p.source_title, d.title) AS document_title,
    COALESCE(p.source_author, a.name) AS author_name,
    p.pin_title,
    p.pin_note,
    p.highlight_text,
    c.chunk_text,
    p.pin_color,
    p.pin_tags,
    p.sort_order,
    p.is_used,
    p.created_at
FROM research_pins p
LEFT JOIN chunks c ON p.chunk_id = c.chunk_id
LEFT JOIN documents d ON COALESCE(p.document_id, c.document_id) = d.document_id
LEFT JOIN authors a ON d.author_id = a.author_id
LEFT JOIN book_projects bp ON p.project_id = bp.project_id
ORDER BY p.project_id, p.sort_order, p.created_at DESC;

-- View: Chat sessions with message counts
CREATE OR REPLACE VIEW v_chat_sessions AS
SELECT
    s.session_id,
    s.title,
    s.project_id,
    bp.project_name,
    s.context_document_id,
    d.title AS context_document_title,
    s.message_count,
    s.is_starred,
    s.is_archived,
    s.created_at,
    s.last_message_at
FROM chat_sessions s
LEFT JOIN book_projects bp ON s.project_id = bp.project_id
LEFT JOIN documents d ON s.context_document_id = d.document_id
WHERE s.is_archived = FALSE
ORDER BY s.is_starred DESC, s.last_message_at DESC NULLS LAST;

-- View: Library health status
CREATE OR REPLACE VIEW v_library_health AS
SELECT
    d.document_id,
    d.title,
    a.name AS author_name,
    d.processing_status,
    tq.quality_grade,
    tq.gibberish_ratio,
    tq.do_not_process,
    CASE
        WHEN d.processing_status = 'failed' THEN 'error'
        WHEN tq.quality_grade IN ('poor', 'very_poor') THEN 'warning'
        WHEN tq.do_not_process = TRUE THEN 'warning'
        WHEN d.processing_status = 'completed' THEN 'healthy'
        ELSE 'pending'
    END AS health_status,
    d.created_at
FROM documents d
LEFT JOIN authors a ON d.author_id = a.author_id
LEFT JOIN text_quality tq ON d.document_id = tq.document_id
ORDER BY
    CASE
        WHEN d.processing_status = 'failed' THEN 1
        WHEN tq.quality_grade IN ('poor', 'very_poor') THEN 2
        WHEN d.processing_status = 'pending' THEN 3
        ELSE 4
    END,
    d.created_at DESC;

-- -----------------------------------------------------------------------------
-- Sample Data (Optional)
-- -----------------------------------------------------------------------------

-- Create a sample "General Research" project for the pinboard
INSERT INTO book_projects (project_name, description, status)
VALUES (
    'General Research',
    'Default project for collecting research materials before assigning to specific books.',
    'active'
)
ON CONFLICT DO NOTHING;

-- =============================================================================
-- MIGRATION COMPLETE
-- =============================================================================
-- Run this migration with:
--   PGPASSWORD='JpGZhjgjNd1M8rrh29BT' psql -h localhost -U research_dev_user \
--     -d research_dev_db -f database/migrations/001_writer_workbench.sql
--
-- Verify with:
--   \dt research_pins
--   \dt chat_history
--   \dt chat_sessions
--   \dt concept_aliases
--   \dt user_preferences
-- =============================================================================
