"""
Observability - Integration with monitoring tools.

This module provides integration with:
- Token Audit: Local usage ledger, MCP queryable
- AI Observer: Real-time dashboard, alerts
- Alert Manager: Stuck agent, cost, error alerts

Usage:
    from agent_orchestrator.observability import (
        TokenAuditClient,
        AIObserverClient,
        AlertManager,
    )

    # Token audit for usage tracking
    audit = TokenAuditClient(mcp_client=mcp, db=db)
    await audit.record_usage(
        provider="anthropic",
        model="claude-3",
        input_tokens=1000,
        output_tokens=200,
        cost_usd=0.01,
    )

    # Alert manager for monitoring
    alerts = AlertManager(db=db)
    await alerts.check_stuck_agent("agent-1", is_stuck=True, stuck_duration_seconds=600)
"""

from .token_audit import (
    TokenAuditClient,
    UsageEntry,
    UsageSummary,
    CostAlert,
    get_token_audit_client,
    set_token_audit_client,
)

from .ai_observer import (
    AIObserverClient,
    MetricPoint,
    AlertRule as ObserverAlertRule,
    Alert as ObserverAlert,
    get_ai_observer_client,
    set_ai_observer_client,
)

from .alerts import (
    AlertManager,
    Alert,
    AlertRule,
    AlertSeverity,
    AlertType,
    AlertState,
    NotificationChannel,
    get_alert_manager,
    set_alert_manager,
)

from .audit import (
    AuditTrail,
    AuditEvent,
    AuditConfig,
    EventType,
    EventSeverity,
    get_audit_trail,
    set_audit_trail,
)

__all__ = [
    # Token Audit
    "TokenAuditClient",
    "UsageEntry",
    "UsageSummary",
    "CostAlert",
    "get_token_audit_client",
    "set_token_audit_client",
    # AI Observer
    "AIObserverClient",
    "MetricPoint",
    "ObserverAlertRule",
    "ObserverAlert",
    "get_ai_observer_client",
    "set_ai_observer_client",
    # Alert Manager
    "AlertManager",
    "Alert",
    "AlertRule",
    "AlertSeverity",
    "AlertType",
    "AlertState",
    "NotificationChannel",
    "get_alert_manager",
    "set_alert_manager",
    # Audit Trail
    "AuditTrail",
    "AuditEvent",
    "AuditConfig",
    "EventType",
    "EventSeverity",
    "get_audit_trail",
    "set_audit_trail",
]
