"""
Merge Gate - Protected branch controls.

This module implements merge controls for protected branches:
- Readiness checks (tests, approvals, risks)
- Conflict detection
- Merge locking (one at a time)

Usage:
    from agent_orchestrator.merge import MergeGate, MergeReadiness

    # Create merge gate
    gate = MergeGate(db=db)
    gate.add_protected_branch("main")

    # Request a merge
    result = await gate.request_merge(
        source_branch="feature/new-feature",
        target_branch="main",
    )

    if result.success:
        print(f"Merged! Commit: {result.merge_commit}")
    else:
        print(f"Failed: {result.error_message}")
"""

from .readiness import (
    MergeReadiness,
    ReadinessReport,
    CheckResult,
    CheckStatus,
    CheckSeverity,
)

from .gate import (
    MergeGate,
    MergeRequest,
    MergeResult,
    MergeStatus,
    MergeStrategy,
    get_merge_gate,
    set_merge_gate,
)

__all__ = [
    # Readiness
    "MergeReadiness",
    "ReadinessReport",
    "CheckResult",
    "CheckStatus",
    "CheckSeverity",
    # Gate
    "MergeGate",
    "MergeRequest",
    "MergeResult",
    "MergeStatus",
    "MergeStrategy",
    "get_merge_gate",
    "set_merge_gate",
]
