"""
Control Loop - Health monitoring, risk gating, and agent management.

This module implements:
- Health monitoring with stuck detection heuristics
- Control loop for agent orchestration
- Risk policy for file/command classification
- Autonomy gate with four-tier decision logic

Key Classes:
- AgentHealthCheck: Stuck detection and health monitoring
- AgentControlLoop: Main orchestration loop
- RiskPolicy: Risk classification for actions
- AutonomyGate: Four-tier autonomy decision engine
"""

from .health import (
    AgentHealthCheck,
    HealthCheckResult,
    StuckReason,
    AgentState,
    StuckDetectionConfig,
    generate_unstick_prompt,
)

from .loop import (
    AgentControlLoop,
    LoopConfig,
    LoopState,
    AgentContext,
)

from .actions import (
    ControlAction,
    ControlActionType,
    ActionResult,
    ActionPolicy,
    EscalationLevel,
    create_reassign_action,
    create_pause_action,
    create_resume_action,
)

from .risk_policy import (
    RiskPolicy,
    RiskLevel,
    RiskClassification,
    get_risk_policy,
    classify_command,
    classify_file,
    is_command_blocked,
    is_file_blocked,
)

from .autonomy_gate import (
    AutonomyGate,
    IntegratedAutonomyGate,
    AutonomyTier,
    ActionType,
    GateDecision,
    AgentAutonomy,
    get_autonomy_gate,
    set_autonomy_gate,
    requires_autonomy_check,
)

__all__ = [
    # Health Check
    "AgentHealthCheck",
    "HealthCheckResult",
    "StuckReason",
    "AgentState",
    "StuckDetectionConfig",
    "generate_unstick_prompt",
    # Control Loop
    "AgentControlLoop",
    "LoopConfig",
    "LoopState",
    "AgentContext",
    # Actions
    "ControlAction",
    "ControlActionType",
    "ActionResult",
    "ActionPolicy",
    "EscalationLevel",
    "create_reassign_action",
    "create_pause_action",
    "create_resume_action",
    # Risk Policy
    "RiskPolicy",
    "RiskLevel",
    "RiskClassification",
    "get_risk_policy",
    "classify_command",
    "classify_file",
    "is_command_blocked",
    "is_file_blocked",
    # Autonomy Gate
    "AutonomyGate",
    "IntegratedAutonomyGate",
    "AutonomyTier",
    "ActionType",
    "GateDecision",
    "AgentAutonomy",
    "get_autonomy_gate",
    "set_autonomy_gate",
    "requires_autonomy_check",
]
