# Known Error Patterns and Fixes

This file documents error patterns encountered by agents and their resolutions.
The Memory Librarian uses this to help agents avoid repeating mistakes.

## Format

Each entry follows this structure:

```markdown
### [Short Title]

**Error Pattern**: What the error looks like
**Cause**: Why this happens
**Fix**: How to resolve it
**Prevention**: How to avoid it
**Tags**: relevant, tags, here
**Source**: Link to commit/issue/task where this was discovered
```

---

## API and Authentication Errors

### Claude API Rate Limit

**Error Pattern**:
```
anthropic.RateLimitError: 429 rate_limit_exceeded
```

**Cause**: Exceeded requests per minute or tokens per minute limit

**Fix**:
1. Wait for rate limit window to reset (usually 60 seconds)
2. Implement exponential backoff
3. Check `Retry-After` header

**Prevention**:
- Use budget controls to pace requests
- Implement request queuing
- Monitor usage_daily table

**Tags**: api, claude, rate-limit, retry
**Source**: Initial implementation

---

### Invalid API Key

**Error Pattern**:
```
anthropic.AuthenticationError: Invalid API key
```

**Cause**: API key not set, expired, or incorrectly configured

**Fix**:
1. Check `.env` file has `ANTHROPIC_API_KEY`
2. Verify key is valid (not expired)
3. Check for extra whitespace in key

**Prevention**:
- Use `config.api_keys.has_anthropic()` before using adapter
- Log helpful error when key missing

**Tags**: api, authentication, configuration
**Source**: Initial implementation

---

## Git and Workspace Errors

### Worktree Already Exists

**Error Pattern**:
```
fatal: 'agent-<id>' is already checked out at '/path/to/agent-<id>'
```

**Cause**: Trying to create a worktree that already exists

**Fix**:
```bash
git worktree prune
git worktree remove /path/to/agent-<id> --force
```

**Prevention**:
- Check `workspace_path.exists()` before creating
- Reuse existing workspace when present

**Tags**: git, worktree, workspace
**Source**: Workspace setup

---

### tmux Session Collision

**Error Pattern**:
```
duplicate session: <agent-id>
```

**Cause**: tmux session with same name already exists

**Fix**:
```bash
tmux has-session -t <agent-id>  # Check if exists
tmux kill-session -t <agent-id>  # Kill if needed
```

**Prevention**:
- Check for existing session before creating
- Use unique agent IDs

**Tags**: tmux, session, workspace
**Source**: Workspace setup

---

## Memory and State Errors

### JSON Decode Error in Project State

**Error Pattern**:
```
json.JSONDecodeError: Expecting value: line 1 column 1
```

**Cause**: `project_state.json` is empty or corrupted

**Fix**:
1. Restore from backup (if available)
2. Or recreate with initial state:
   ```python
   journal.write_state({"version": "1.0.0", "objectives": []})
   ```

**Prevention**:
- Always validate JSON before writing
- Keep backups of project_state.json

**Tags**: json, state, corruption
**Source**: Project journal

---

### Memory Patch Rejected - Secrets Detected

**Error Pattern**:
```
MemoryWriteGate: Patch rejected - secrets detected in content
```

**Cause**: Agent tried to write content containing API keys or passwords

**Fix**:
1. DO NOT approve the patch
2. Review the source task for secret handling issues
3. Clean up any temp files

**Prevention**:
- Run `SecretRedactor.redact()` on all agent output
- Use `SecretFileGuard` for file operations

**Tags**: secrets, memory, security
**Source**: Memory Write Gate

---

## Test and CI Errors

### pytest Collection Error

**Error Pattern**:
```
ERROR collecting tests/...
ModuleNotFoundError: No module named 'agent_orchestrator'
```

**Cause**: Package not installed in development mode

**Fix**:
```bash
pip install -e .
```

**Prevention**:
- Include in setup instructions
- Add to CI pipeline

**Tags**: pytest, import, installation
**Source**: Test setup

---

## Adding New Entries

When you encounter a new error pattern:

1. Document it following the format above
2. Add appropriate tags for searchability
3. Include the source (commit, issue, or task ID)
4. Submit as a memory patch:
   ```
   PATCH: ADD error-fixes.md entry
   RISK: LOW
   EVIDENCE: [link to task]
   ```

The Memory Librarian will merge similar entries periodically.
