#!/usr/bin/env python3
"""
Simple Task Example

Demonstrates how to submit a single task to the orchestrator
and wait for completion.

Usage:
    python examples/basic/simple_task.py
"""

import asyncio
import logging

from agent_orchestrator import Orchestrator

# Configure logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


async def main():
    """Submit a simple task and wait for completion."""

    # Initialize the orchestrator
    orchestrator = Orchestrator()

    try:
        # Start the orchestrator
        logger.info("Starting orchestrator...")
        await orchestrator.start()

        # Submit a task
        logger.info("Submitting task...")
        task = await orchestrator.submit_task(
            description="Create a Python function that checks if a number is prime. "
            "Include docstring and type hints.",
        )
        logger.info(f"Task submitted: {task.id}")

        # Wait for completion
        logger.info("Waiting for task completion...")
        result = await orchestrator.wait_for_task(task.id, timeout=300)

        # Display results
        logger.info(f"Task Status: {result.status}")
        if result.status == "completed":
            logger.info(f"Agent: {result.agent_id}")
            logger.info(f"Tokens Used: {result.tokens_used}")
            logger.info(f"Cost: ${result.cost_usd:.4f}")
            logger.info(f"\nOutput:\n{result.output}")
        else:
            logger.error(f"Task failed: {result.error}")

    finally:
        # Clean shutdown
        logger.info("Shutting down orchestrator...")
        await orchestrator.shutdown()


if __name__ == "__main__":
    asyncio.run(main())
