#!/usr/bin/env python3
"""
List Agents Example

Demonstrates how to list all available agents and their status.

Usage:
    python examples/basic/list_agents.py
"""

import asyncio
import logging

from agent_orchestrator import Orchestrator

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


async def main():
    """List all agents and their current status."""

    orchestrator = Orchestrator()

    try:
        await orchestrator.start()

        # Get all agents
        agents = await orchestrator.list_agents()

        print("\n" + "=" * 60)
        print("AVAILABLE AGENTS")
        print("=" * 60)

        for agent in agents:
            status_emoji = {
                "available": "🟢",
                "busy": "🟡",
                "exhausted": "🔴",
                "offline": "⚫",
            }.get(agent.status, "❓")

            print(f"\n{status_emoji} {agent.agent_id}")
            print(f"   Type: {agent.agent_type}")
            print(f"   Status: {agent.status}")
            print(f"   Tier: {agent.subscription_tier}")
            if agent.current_task:
                print(f"   Current Task: {agent.current_task}")
            if agent.capabilities:
                print(f"   Capabilities: {', '.join(agent.capabilities)}")

        print("\n" + "=" * 60)
        print(f"Total: {len(agents)} agents")
        print(
            f"Available: {sum(1 for a in agents if a.status == 'available')}"
        )
        print("=" * 60 + "\n")

    finally:
        await orchestrator.shutdown()


if __name__ == "__main__":
    asyncio.run(main())
