# Configuration Reference

Complete reference for all Agent Orchestrator configuration options.

## Environment Variables

### Core Settings

| Variable | Default | Description |
|----------|---------|-------------|
| `DATABASE_URL` | `sqlite:///./orchestrator.db` | Database connection string |
| `LOG_LEVEL` | `INFO` | Logging level (DEBUG, INFO, WARNING, ERROR) |
| `WORKSPACE_DIR` | `./workspaces` | Directory for agent workspaces |

### API Keys (Optional)

```bash
# Anthropic (for Claude SDK)
ANTHROPIC_API_KEY=sk-ant-...

# OpenAI (for OpenAI Agents SDK)
OPENAI_API_KEY=sk-...

# Google (for Gemini API)
GOOGLE_API_KEY=AIza...
```

### Budget Settings

| Variable | Default | Description |
|----------|---------|-------------|
| `DAILY_BUDGET_USD` | `50.00` | Total daily budget |
| `ALERT_THRESHOLD_50` | `true` | Alert at 50% usage |
| `ALERT_THRESHOLD_75` | `true` | Alert at 75% usage |
| `ALERT_THRESHOLD_90` | `true` | Alert at 90% usage |
| `BLOCK_ON_EXHAUSTION` | `true` | Block tasks when budget exhausted |

### Observability

| Variable | Default | Description |
|----------|---------|-------------|
| `SLACK_WEBHOOK_URL` | None | Slack webhook for alerts |
| `WEBHOOK_URL` | None | Generic webhook URL |
| `ENABLE_TRACING` | `true` | Enable session tracing |
| `TRACE_STORAGE` | `sqlite` | Trace storage backend |

### API Server

| Variable | Default | Description |
|----------|---------|-------------|
| `API_HOST` | `0.0.0.0` | API server host |
| `API_PORT` | `8080` | API server port |
| `API_KEY` | None | API authentication key |
| `CORS_ORIGINS` | `*` | Allowed CORS origins |

## Configuration Files

### config.yaml

Main configuration file:

```yaml
# config.yaml
orchestrator:
  name: "agent-orchestrator"
  log_level: INFO
  workspace_dir: ./workspaces

database:
  url: sqlite:///./orchestrator.db
  pool_size: 5
  echo: false

agents:
  health_check_interval_seconds: 30
  stuck_threshold_seconds: 300
  max_concurrent_tasks: 5

  defaults:
    timeout_seconds: 300
    auto_prompt_enabled: true
    auto_prompt_delay_seconds: 60

  claude_code:
    enabled: true
    command: claude-code
    subscription_tier: max
    capabilities:
      - CODE_EDIT
      - CODE_REVIEW
      - FILE_WRITE
      - GIT
    budget:
      daily_usd: 20.00
      daily_tokens: 500000

  gemini_cli:
    enabled: true
    command: gemini
    subscription_tier: pro
    capabilities:
      - SEARCH
      - LARGE_CONTEXT
      - CODE_REVIEW
    budget:
      daily_usd: 15.00
      daily_tokens: 1000000

  codex_cli:
    enabled: true
    command: codex
    subscription_tier: plus
    capabilities:
      - RUN_TESTS
      - TERMINAL
      - AUTONOMOUS
    budget:
      daily_usd: 15.00
      daily_tokens: 300000

risk:
  default_level: MEDIUM
  auto_approve_low: true
  require_approval_high: true
  block_critical: true

  patterns:
    - pattern: "rm -rf"
      level: CRITICAL
    - pattern: "git push --force"
      level: CRITICAL
    - pattern: "npm install"
      level: MEDIUM
    - pattern: "git commit"
      level: LOW

  blocked_paths:
    - "/.env"
    - "/secrets/"
    - "/.ssh/"

budget:
  daily_limit_usd: 50.00

  alerts:
    - threshold: 0.50
      severity: INFO
      message: "50% of daily budget used"
    - threshold: 0.75
      severity: WARNING
      message: "75% of daily budget used"
    - threshold: 0.90
      severity: ERROR
      message: "90% of daily budget used"
    - threshold: 1.00
      severity: CRITICAL
      message: "Daily budget exhausted"
      block: true

  mcp_servers:
    filesystem:
      daily_calls: 1000
    github:
      daily_calls: 500
    database:
      daily_calls: 200

observability:
  alerts:
    slack:
      enabled: true
      webhook_url: ${SLACK_WEBHOOK_URL}
      channel: "#agent-alerts"
    webhook:
      enabled: false
      url: ${WEBHOOK_URL}

  tracing:
    enabled: true
    storage: sqlite
    retention_days: 30

api:
  host: 0.0.0.0
  port: 8080
  api_key: ${API_KEY}
  cors_origins:
    - "*"
  rate_limit:
    requests_per_minute: 100
```

### risk_policy.yaml

Custom risk classification rules:

```yaml
# risk_policy.yaml
risk_policy:
  version: "1.0"

  command_patterns:
    # Critical - Auto-reject
    - pattern: "rm -rf /"
      level: CRITICAL
      description: "Delete root filesystem"
    - pattern: "git push.*--force"
      level: CRITICAL
      description: "Force push"
    - pattern: "DROP DATABASE"
      level: CRITICAL
      description: "Drop database"

    # High - Require explicit approval
    - pattern: "terraform destroy"
      level: HIGH
      description: "Destroy infrastructure"
    - pattern: "kubectl delete"
      level: HIGH
      description: "Delete Kubernetes resources"
    - pattern: "ALTER TABLE"
      level: HIGH
      description: "Alter database schema"

    # Medium - Confirm before execution
    - pattern: "npm install"
      level: MEDIUM
      description: "Install npm packages"
    - pattern: "git push"
      level: MEDIUM
      description: "Push to remote"
    - pattern: "docker build"
      level: MEDIUM
      description: "Build Docker image"

    # Low - Auto-approve
    - pattern: "git status"
      level: LOW
      description: "Check git status"
    - pattern: "pytest"
      level: LOW
      description: "Run tests"
    - pattern: "cat "
      level: LOW
      description: "Read file"

  file_patterns:
    # Critical files
    - pattern: ".*\\.env$"
      level: CRITICAL
      description: "Environment file"
    - pattern: ".*/secrets/.*"
      level: CRITICAL
      description: "Secrets directory"

    # High-risk files
    - pattern: ".*/auth/.*"
      level: HIGH
      description: "Authentication code"
    - pattern: ".*/migrations/.*"
      level: HIGH
      description: "Database migrations"

    # Medium-risk files
    - pattern: ".*/config/.*"
      level: MEDIUM
      description: "Configuration files"

  overrides:
    # Override specific patterns
    - pattern: "npm install --save-dev"
      level: LOW
      description: "Dev dependencies are safe"

  allowed_paths:
    - /src/
    - /tests/
    - /docs/

  blocked_paths:
    - /.env
    - /secrets/
    - /.ssh/
    - /node_modules/
```

### workflows/

JSON workflow definitions:

```json
// workflows/feature-implementation.json
{
  "name": "feature-implementation",
  "version": "1.0",
  "description": "Standard feature implementation workflow",
  "variables": {
    "feature_name": "",
    "branch_prefix": "feature/"
  },
  "timeout_minutes": 120,
  "on_failure": "stop",
  "max_concurrent_steps": 3,
  "steps": [
    {
      "id": "create-branch",
      "type": "task",
      "agent_id": "claude-code",
      "prompt": "Create a new branch: ${branch_prefix}${feature_name}",
      "timeout_seconds": 60
    },
    {
      "id": "design",
      "type": "task",
      "agent_id": "claude-code",
      "prompt": "Design the architecture for ${feature_name}",
      "output_var": "design_doc"
    },
    {
      "id": "review-design",
      "type": "approval",
      "message": "Please review the design for ${feature_name}",
      "timeout_seconds": 3600
    },
    {
      "id": "implement",
      "type": "parallel",
      "steps": [
        {
          "id": "backend",
          "type": "task",
          "agent_id": "claude-code",
          "prompt": "Implement backend for ${feature_name}"
        },
        {
          "id": "frontend",
          "type": "task",
          "agent_id": "gemini-cli",
          "prompt": "Implement frontend for ${feature_name}"
        }
      ]
    },
    {
      "id": "test",
      "type": "task",
      "agent_id": "codex-cli",
      "prompt": "Write and run tests for ${feature_name}",
      "dependencies": ["implement"]
    }
  ]
}
```

## Agent Configuration

### Capabilities

Available agent capabilities:

| Capability | Description |
|------------|-------------|
| `FILE_READ` | Read files |
| `FILE_WRITE` | Write/create files |
| `FILE_DELETE` | Delete files |
| `CODE_EDIT` | Edit existing code |
| `CODE_REVIEW` | Review code quality |
| `CODE_REFACTOR` | Refactor code |
| `CODE_GENERATE` | Generate new code |
| `RUN_TESTS` | Execute tests |
| `DEBUG` | Debug issues |
| `GIT` | Git operations |
| `TERMINAL` | Execute commands |
| `DEPLOY` | Deploy code |
| `SEARCH` | Search capabilities |
| `STREAMING` | Streaming responses |
| `FUNCTION_CALLING` | Tool/function calling |
| `FAST` | Low latency |
| `LARGE_CONTEXT` | Large context window |
| `AUTONOMOUS` | Can work autonomously |

### Subscription Tiers

| Provider | Tier | Rate Limit | Context |
|----------|------|------------|---------|
| Anthropic | Free | 5/min | 100K |
| Anthropic | Pro | 50/min | 200K |
| Anthropic | Max | 100/min | 200K |
| OpenAI | Free | 3/min | 8K |
| OpenAI | Plus | 40/min | 128K |
| OpenAI | Pro | 100/min | 200K |
| Google | Free | 15/min | 32K |
| Google | Pro | 60/min | 1M |
| Google | Ultra | 120/min | 2M |

## Isolation Configuration

### Tmux (Default)

```yaml
isolation:
  type: tmux
  session_prefix: agent_
  window_size: "200x50"
```

### Docker

```yaml
isolation:
  type: docker
  image: python:3.11-slim
  network_mode: bridge
  resource_limits:
    memory_mb: 4096
    cpu_cores: 2.0
    pids_limit: 100
  credentials:
    inject_git_config: true
    inject_ssh_keys: true
    inject_aws_credentials: false
```

### Podman

```yaml
isolation:
  type: podman
  image: python:3.11-slim
  rootless: true
  resource_limits:
    memory_mb: 4096
    cpu_cores: 2.0
```

## Validation

Validate configuration:

```bash
# Validate config file
python -m agent_orchestrator config validate

# Validate specific file
python -m agent_orchestrator config validate --file config.yaml

# Show effective configuration
python -m agent_orchestrator config show
```

## Environment-Specific Configuration

Use environment prefixes:

```bash
# Development
ORCHESTRATOR_LOG_LEVEL=DEBUG
ORCHESTRATOR_DAILY_BUDGET_USD=10.00

# Production
ORCHESTRATOR_LOG_LEVEL=INFO
ORCHESTRATOR_DAILY_BUDGET_USD=100.00
```

Or use separate config files:

```bash
# Development
python -m agent_orchestrator --config config.dev.yaml

# Production
python -m agent_orchestrator --config config.prod.yaml
```
